<?php
// 📁 Fichier: auth/verify_token.php
// 🛠️ Système d'authentification pour ONL Dashboard

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Gestion des requêtes OPTIONS (CORS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Configuration de la base de données
require_once dirname(__DIR__) . '/config/database.php';

/**
 * Vérifie et valide le token d'authentification
 * @return array Informations utilisateur ou erreur
 */
function verifyAuthToken() {
    try {
        // 1. Récupération du token
        $token = null;
        
        // Vérification dans l'en-tête Authorization
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                $token = $matches[1];
            }
        }
        
        // Vérification dans les paramètres GET/POST
        if (!$token && isset($_REQUEST['token'])) {
            $token = $_REQUEST['token'];
        }
        
        // Vérification dans la session
        if (!$token && isset($_SESSION['auth_token'])) {
            $token = $_SESSION['auth_token'];
        }
        
        if (!$token) {
            return [
                'success' => false,
                'error' => 'Token d\'authentification manquant',
                'code' => 401
            ];
        }
        
        // 2. Validation du token
        $userData = validateToken($token);
        
        if (!$userData) {
            return [
                'success' => false,
                'error' => 'Token invalide ou expiré',
                'code' => 401
            ];
        }
        
        // 3. Vérification des permissions
        if (!checkUserPermissions($userData)) {
            return [
                'success' => false,
                'error' => 'Permissions insuffisantes',
                'code' => 403
            ];
        }
        
        return [
            'success' => true,
            'user' => $userData,
            'permissions' => getUserPermissions($userData)
        ];
        
    } catch (Exception $e) {
        error_log("Erreur vérification token: " . $e->getMessage());
        
        return [
            'success' => false,
            'error' => 'Erreur interne du serveur',
            'code' => 500
        ];
    }
}

/**
 * Valide le token JWT ou token personnalisé
 */
function validateToken($token) {
    global $pdo;
    
    try {
        // Méthode 1: Vérification directe par email/session (pour la compatibilité)
        // Si le token ressemble à un email, on cherche l'utilisateur
        if (filter_var($token, FILTER_VALIDATE_EMAIL)) {
            $stmt = $pdo->prepare("
                SELECT * FROM utilisateurs 
                WHERE email = ? AND actif = 1
            ");
            $stmt->execute([$token]);
            $userData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($userData) {
                return $userData;
            }
        }
        
        // Méthode 2: Token de session simple
        // Pour une authentification basique, on peut utiliser une combinaison email+hash
        $stmt = $pdo->prepare("
            SELECT * FROM utilisateurs 
            WHERE MD5(CONCAT(email, id, DATE(created_at))) = ? AND actif = 1
        ");
        $stmt->execute([$token]);
        $userData = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($userData) {
            return $userData;
        }
        
        // Méthode 3: JWT Token (si vous utilisez JWT)
        if (function_exists('validateJWT')) {
            return validateJWT($token);
        }
        
        // Méthode 4: Token basé sur l'ID utilisateur (pour les tests)
        if (is_numeric($token)) {
            $stmt = $pdo->prepare("
                SELECT * FROM utilisateurs 
                WHERE id = ? AND actif = 1
            ");
            $stmt->execute([$token]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        // Méthode 5: Recherche par hash du mot de passe (fallback)
        $stmt = $pdo->prepare("
            SELECT * FROM utilisateurs 
            WHERE mot_de_passe = ? AND actif = 1
        ");
        $stmt->execute([$token]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
        
    } catch (Exception $e) {
        error_log("Erreur validation token: " . $e->getMessage());
        return false;
    }
}

/**
 * Vérifie les permissions utilisateur
 */
function checkUserPermissions($userData) {
    // Vérification basique
    if (!$userData || !isset($userData['id'])) {
        return false;
    }
    
    // Vérification du statut actif
    if (isset($userData['actif']) && $userData['actif'] != 1) {
        return false;
    }
    
    // Vérification du rôle (selon votre enum)
    $allowedRoles = ['admin', 'superviseur', 'controleur', 'referent'];
    if (isset($userData['role']) && !in_array($userData['role'], $allowedRoles)) {
        return false;
    }
    
    return true;
}

/**
 * Récupère les permissions utilisateur selon les rôles ONL
 */
function getUserPermissions($userData) {
    $role = $userData['role'] ?? 'referent';
    
    $permissions = [
        'admin' => ['read', 'write', 'delete', 'export', 'manage_users', 'manage_sites', 'view_all_sites'],
        'superviseur' => ['read', 'write', 'export', 'manage_site', 'view_all_sites'],
        'controleur' => ['read', 'write', 'export', 'manage_site'],
        'referent' => ['read', 'write', 'export']
    ];
    
    return $permissions[$role] ?? ['read'];
}

/**
 * Fonction JWT personnalisée (optionnelle)
 */
function validateJWT($token) {
    // Implémentation JWT simple
    // Remplacez par votre propre logique JWT si nécessaire
    
    $secretKey = 'votre_cle_secrete_jwt'; // À changer !
    
    try {
        // Décodage du token JWT
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return false;
        }
        
        $header = json_decode(base64_decode($parts[0]), true);
        $payload = json_decode(base64_decode($parts[1]), true);
        $signature = $parts[2];
        
        // Vérification de la signature
        $expectedSignature = base64_encode(hash_hmac('sha256', $parts[0] . '.' . $parts[1], $secretKey, true));
        
        if ($signature !== rtrim($expectedSignature, '=')) {
            return false;
        }
        
        // Vérification de l'expiration
        if (isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }
        
        return $payload;
        
    } catch (Exception $e) {
        return false;
    }
}

// Exécution de la vérification si ce fichier est appelé directement
if (!isset($INCLUDED_VERIFY_TOKEN)) {
    $authResult = verifyAuthToken();
    
    if (!$authResult['success']) {
        http_response_code($authResult['code']);
        echo json_encode($authResult);
        exit();
    }
    
    // Variables globales pour les autres scripts
    $currentUser = $authResult['user'];
    $userPermissions = $authResult['permissions'];
}

// Fonction d'inclusion sécurisée pour les autres scripts
function requireAuth() {
    global $INCLUDED_VERIFY_TOKEN;
    $INCLUDED_VERIFY_TOKEN = true;
    
    $authResult = verifyAuthToken();
    
    if (!$authResult['success']) {
        http_response_code($authResult['code']);
        echo json_encode([
            'success' => false,
            'error' => $authResult['error']
        ]);
        exit();
    }
    
    return $authResult;
}

// Configuration pour éviter les inclusions multiples
$INCLUDED_VERIFY_TOKEN = true;
?>