<?php
class GeoValidation {
    private const MAX_DISTANCE_METERS = 20;
    
    public static function validateProximityToBorne($currentLat, $currentLon, $borneLat, $borneLon) {
        if (!$borneLat || !$borneLon) {
            return ['valid' => false, 'error' => 'Coordonnées de référence manquantes'];
        }
        
        $distance = self::calculateDistance($currentLat, $currentLon, $borneLat, $borneLon);
        
        return [
            'valid' => $distance <= self::MAX_DISTANCE_METERS,
            'distance' => round($distance, 2),
            'max_allowed' => self::MAX_DISTANCE_METERS,
            'error' => $distance > self::MAX_DISTANCE_METERS ? 
                "Distance trop élevée: {$distance}m (max: " . self::MAX_DISTANCE_METERS . "m)" : null
        ];
    }
    
    public static function calculateDistance($lat1, $lon1, $lat2, $lon2) {
        $earthRadius = 6371000; // mètres
        
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        
        $a = sin($dLat/2) * sin($dLat/2) + 
             cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * 
             sin($dLon/2) * sin($dLon/2);
        
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        
        return $earthRadius * $c;
    }
    
    public static function getOrCreateBorneReference($site_id, $pdo, $lat = null, $lon = null) {
        // Récupérer borne existante
        $stmt = $pdo->prepare("
            SELECT latitude_reference, longitude_reference 
            FROM profils_topographiques 
            WHERE site_id = ? AND latitude_reference IS NOT NULL 
            LIMIT 1
        ");
        $stmt->execute([$site_id]);
        $borne = $stmt->fetch();
        
        if ($borne) {
            return [
                'latitude' => (float)$borne['latitude_reference'],
                'longitude' => (float)$borne['longitude_reference'],
                'is_new' => false
            ];
        }
        
        // Si pas de borne et nouvelles coordonnées fournies
        if ($lat && $lon) {
            return [
                'latitude' => (float)$lat,
                'longitude' => (float)$lon,
                'is_new' => true
            ];
        }
        
        return null;
    }
}
?>