<?php
// =======================================================================
// FICHIER: dechets/create.php (ADAPTÉ À VOTRE BASE EXISTANTE)
// =======================================================================

// Inclure la classe Database
require_once '../config/database.php';

// Connexion à la base de données
$database = new Database();
$pdo = $database->getConnection();

if (!$pdo) {
    jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
}

// Lire les données JSON
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    jsonResponse(false, 'Données JSON invalides', null, 400);
}

// =======================================================================
// GESTION DU MODE DEBUG
// =======================================================================

$skipDistanceValidation = isset($input['skip_distance_validation']) && $input['skip_distance_validation'] === true;
$debugMode = isset($input['debug_mode']) && $input['debug_mode'] === true;
$forceSave = isset($input['force_save']) && $input['force_save'] === true;

if ($debugMode || $skipDistanceValidation || $forceSave) {
    error_log("Mode debug activé - Flags: skip=" . ($skipDistanceValidation ? 'true' : 'false') . 
             ", debug=" . ($debugMode ? 'true' : 'false') . 
             ", force=" . ($forceSave ? 'true' : 'false'));
}

// =======================================================================
// VALIDATION DES DONNÉES D'ENTRÉE
// =======================================================================

$required_fields = [
    'code_referent', 'site_id', 'date_releve', 'heure_debut',
    'latitude_debut', 'longitude_debut', 'altitude_debut',
    'latitude_fin', 'longitude_fin', 'dechets'
];

foreach ($required_fields as $field) {
    if (!isset($input[$field]) || (is_string($input[$field]) && trim($input[$field]) === '')) {
        jsonResponse(false, "Champ requis manquant: $field", null, 400);
    }
}

// Extraction des données
$code_referent = trim($input['code_referent']);
$site_id = (int)$input['site_id'];
$date_releve = $input['date_releve'];
$heure_debut = $input['heure_debut'];
$latitude_debut = (float)$input['latitude_debut'];
$longitude_debut = (float)$input['longitude_debut'];
$altitude_debut = (float)$input['altitude_debut'];
$latitude_fin = (float)$input['latitude_fin'];
$longitude_fin = (float)$input['longitude_fin'];
$orientation_plage = $input['orientation_plage'] ?? '01';
$courant_dominant = $input['courant_dominant'] ?? null;
$vent_dominant = $input['vent_dominant'] ?? null;
$evenements_extremes = $input['evenements_extremes'] ?? '["9"]';
$autre_evenement = $input['autre_evenement'] ?? '';
$statut = $input['statut'] ?? 'brouillon';
$dechets = $input['dechets'];

// Déterminer l'utilisateur_id (obligatoire dans votre base)
$utilisateur_id = 3; // BABACAR SAMB par défaut, à adapter selon votre système d'auth

// Validation des coordonnées GPS
if ($latitude_debut < -90 || $latitude_debut > 90 || 
    $longitude_debut < -180 || $longitude_debut > 180) {
    jsonResponse(false, 'Coordonnées GPS début invalides', null, 400);
}

if ($latitude_fin < -90 || $latitude_fin > 90 || 
    $longitude_fin < -180 || $longitude_fin > 180) {
    jsonResponse(false, 'Coordonnées GPS fin invalides', null, 400);
}

// Validation des déchets
if (!is_array($dechets) || empty($dechets)) {
    jsonResponse(false, 'Données de déchets manquantes ou invalides', null, 400);
}

// =======================================================================
// GESTION DES BORNES DE RÉFÉRENCE (SIMULATION - pas de table dédiée)
// =======================================================================

// Récupérer la dernière borne de référence depuis releves_dechets existants
$borne_lat = $latitude_debut;
$borne_lon = $longitude_debut;

try {
    // Chercher un relevé existant pour ce site comme référence
    $stmt = $pdo->prepare("
        SELECT latitude_debut, longitude_debut 
        FROM releves_dechets 
        WHERE site_id = ? 
        AND latitude_debut IS NOT NULL 
        AND longitude_debut IS NOT NULL
        ORDER BY created_at ASC 
        LIMIT 1
    ");
    $stmt->execute([$site_id]);
    $existing_releve = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existing_releve) {
        $borne_lat = (float)$existing_releve['latitude_debut'];
        $borne_lon = (float)$existing_releve['longitude_debut'];
        error_log("Borne de référence trouvée depuis relevé existant pour site $site_id: $borne_lat, $borne_lon");
    } else {
        error_log("Premier relevé pour site $site_id - nouvelle borne de référence: $borne_lat, $borne_lon");
    }
} catch(PDOException $e) {
    error_log("Erreur récupération borne référence: " . $e->getMessage());
}

// =======================================================================
// VALIDATION DE DISTANCE
// =======================================================================

function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371000; // mètres
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat/2) * sin($dLat/2) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * sin($dLon/2) * sin($dLon/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    return $earthRadius * $c;
}

if (!$skipDistanceValidation && !$debugMode && !$forceSave) {
    $distance_debut = calculateDistance($latitude_debut, $longitude_debut, $borne_lat, $borne_lon);
    $distance_fin = calculateDistance($latitude_fin, $longitude_fin, $borne_lat, $borne_lon);
    
    if ($distance_debut > 20) {
        jsonResponse(false, 'Position début: Distance trop élevée: ' . round($distance_debut, 2) . 'm (max: 20m)', [
            'debug_info' => [
                'distance_calculated' => round($distance_debut, 2),
                'max_allowed' => 20
            ]
        ], 400);
    }
    
    if ($distance_fin > 20) {
        jsonResponse(false, 'Position fin: Distance trop élevée: ' . round($distance_fin, 2) . 'm (max: 20m)', [
            'debug_info' => [
                'distance_calculated' => round($distance_fin, 2),
                'max_allowed' => 20
            ]
        ], 400);
    }
    
    error_log("Validation de distance OK - Début: " . round($distance_debut, 2) . "m, Fin: " . round($distance_fin, 2) . "m");
} else {
    $distance_debut = calculateDistance($latitude_debut, $longitude_debut, $borne_lat, $borne_lon);
    $distance_fin = calculateDistance($latitude_fin, $longitude_fin, $borne_lat, $borne_lon);
    
    error_log("MODE DEBUG: Validation de distance ignorée - Début: " . round($distance_debut, 2) . "m, Fin: " . round($distance_fin, 2) . "m");
}

// =======================================================================
// INSERTION EN BASE DE DONNÉES
// =======================================================================

try {
    $pdo->beginTransaction();
    
    // Insertion du relevé principal (AVEC utilisateur_id requis, altitude_borne = altitude_debut)
    $stmt = $pdo->prepare("
        INSERT INTO releves_dechets (
            code_referent, utilisateur_id, site_id, date_releve, heure_debut,
            latitude_debut, longitude_debut, altitude_debut,
            latitude_fin, longitude_fin, altitude_borne,
            orientation_plage, courant_dominant, vent_dominant,
            evenements_extremes, autre_evenement, statut,
            created_at
        ) VALUES (
            ?, ?, ?, ?, ?,
            ?, ?, ?,
            ?, ?, ?,
            ?, ?, ?,
            ?, ?, ?,
            NOW()
        )
    ");
    
    $stmt->execute([
        $code_referent, $utilisateur_id, $site_id, $date_releve, $heure_debut,
        $latitude_debut, $longitude_debut, $altitude_debut,
        $latitude_fin, $longitude_fin, $altitude_debut, // altitude_borne = altitude_debut
        $orientation_plage, $courant_dominant, $vent_dominant,
        $evenements_extremes, $autre_evenement, $statut
    ]);
    
    $releve_id = $pdo->lastInsertId();
    
    // Insertion des déchets dans dechets_detail (nom existant dans votre base)
    $stmt_dechet = $pdo->prepare("
        INSERT INTO dechets_detail (
            releve_id, categorie, nombre_articles, poids,
            source_pollution, photo, commentaires,
            has_photo, photo_filename
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $dechets_count = 0;
    $photos_count = 0;
    
    foreach ($dechets as $dechet) {
        $categorie = $dechet['categorie'] ?? 'PL';
        $nombre_articles = (int)($dechet['nombre_articles'] ?? 0);
        $poids = (float)($dechet['poids'] ?? 0.0);
        $source_pollution = $dechet['source_pollution'] ?? 'autres';
        $photo_base64 = $dechet['photo_base64'] ?? null;
        $commentaires = $dechet['commentaires'] ?? '';
        
        // Conversion des codes de source de pollution pour correspondre à votre base
        $source_code = '5'; // Autres par défaut
        switch($source_pollution) {
            case 'menagers': $source_code = '1'; break;
            case 'industriels': $source_code = '2'; break;
            case 'peche': $source_code = '3'; break;
            case 'sanitaires': $source_code = '4'; break;
            default: $source_code = '5'; break;
        }
        
        // Gestion des photos (conversion base64 en BLOB)
        $photo_blob = null;
        $has_photo = 0;
        $photo_filename = null;
        
        if (!empty($photo_base64)) {
            $photo_blob = base64_decode($photo_base64);
            $has_photo = 1;
            $photo_filename = 'photo_' . $releve_id . '_' . $categorie . '_' . time() . '.jpg';
            $photos_count++;
        }
        
        // Insérer seulement si il y a des données significatives
        if ($nombre_articles > 0 || $poids > 0.0 || $has_photo) {
            $stmt_dechet->execute([
                $releve_id, $categorie, $nombre_articles, $poids,
                $source_code, $photo_blob, $commentaires,
                $has_photo, $photo_filename
            ]);
            
            $dechets_count++;
        }
    }
    
    $pdo->commit();
    
    // Réponse de succès
    jsonResponse(true, 'Relevé de déchets créé avec succès', [
        'releve_id' => $releve_id,
        'id' => $releve_id,
        'dechets_count' => $dechets_count,
        'photos_count' => $photos_count,
        'utilisateur_id' => $utilisateur_id,
        'site_id' => $site_id,
        'date_releve' => $date_releve,
        'heure_debut' => $heure_debut,
        'debug_mode' => $debugMode || $skipDistanceValidation || $forceSave,
        'borne_reference' => [
            'latitude' => $borne_lat,
            'longitude' => $borne_lon,
            'is_new' => false,
            'distance_from_borne' => 0
        ]
    ]);
    
    error_log("✅ Relevé déchets créé avec succès: ID=$releve_id, Déchets=$dechets_count, Photos=$photos_count, Utilisateur=$utilisateur_id");
    
} catch(PDOException $e) {
    $pdo->rollback();
    
    error_log("❌ Erreur création relevé déchets: " . $e->getMessage());
    
    jsonResponse(false, 'Erreur lors de la création du relevé de déchets', [
        'error' => $e->getMessage(),
        'debug_info' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'sql_error' => $e->getCode()
        ]
    ], 500);
}

?>