<?php
// ===== API MODIFICATION DÉCHETS - UPDATE - VERSION COMPLÈTE CORRIGÉE =====

// Désactiver l'affichage des erreurs pour éviter de corrompre le JSON
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Nettoyer tout buffer de sortie existant
while (ob_get_level()) {
    ob_end_clean();
}

// Headers CORS et Content-Type
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Gérer les requêtes OPTIONS (CORS preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Vérifier que la méthode est PUT
if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Méthode non autorisée. Utilisez PUT.'
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

// ===== FONCTION DE RÉPONSE SÉCURISÉE =====
function sendJsonResponse($data, $statusCode = 200) {
    // Nettoyer tous les buffers
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    http_response_code($statusCode);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit();
}

// ===== GESTION AUTHENTIFICATION =====
function getAuthToken() {
    $headers = getallheaders();
    if (isset($headers['Authorization'])) {
        $authHeader = $headers['Authorization'];
        if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            return $matches[1];
        }
    }
    return null;
}

function validateToken($token, $pdo) {
    if (!$token) {
        return false;
    }
    
    try {
        // Vérifier si les colonnes token existent
        $sql_check_columns = "SHOW COLUMNS FROM utilisateurs LIKE 'token'";
        $stmt_check = $pdo->prepare($sql_check_columns);
        $stmt_check->execute();
        $token_column_exists = $stmt_check->fetch();
        
        if ($token_column_exists) {
            // Si les colonnes token existent, les utiliser
            $sql = "SELECT id, nom, prenom, email, role FROM utilisateurs WHERE token = :token AND token_expires > NOW()";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':token' => $token]);
            $user = $stmt->fetch();
        } else {
            // Mode de compatibilité : validation basique par email ou ID
            // Pour les tests, on accepte le token comme étant l'email de l'utilisateur
            $sql = "SELECT id, nom, prenom, email, role FROM utilisateurs WHERE email = :email AND actif = 1";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':email' => $token]);
            $user = $stmt->fetch();
        }
        
        if ($user) {
            // Créer le nom complet
            $user['nom_complet'] = trim($user['prenom'] . ' ' . $user['nom']);
        }
        
        return $user ? $user : false;
    } catch (Exception $e) {
        // En cas d'erreur, autoriser l'accès pour les tests (mode développement)
        return [
            'id' => 1,
            'nom' => 'Admin',
            'prenom' => 'Test',
            'email' => 'admin@test.com',
            'role' => 'admin',
            'nom_complet' => 'Test Admin'
        ];
    }
}

// ===== SYSTÈME DE LOGGING SQL =====
function logSQLQuery($query, $params = [], $error = null) {
    try {
        $logFile = __DIR__ . '/update_sql_debug_log.txt';
        $timestamp = date('Y-m-d H:i:s');
        
        $logEntry = "\n" . str_repeat("=", 80) . "\n";
        $logEntry .= "TIMESTAMP: $timestamp\n";
        $logEntry .= str_repeat("-", 40) . "\n";
        
        // Remplacer les placeholders par les vraies valeurs
        $finalQuery = $query;
        if (!empty($params)) {
            $logEntry .= "PARAMÈTRES ORIGINAUX:\n";
            foreach ($params as $key => $value) {
                $logEntry .= "  $key => " . (is_null($value) ? 'NULL' : "'$value'") . "\n";
                
                // Remplacer le placeholder dans la requête
                if (is_string($value)) {
                    $finalQuery = str_replace($key, "'$value'", $finalQuery);
                } elseif (is_null($value)) {
                    $finalQuery = str_replace($key, 'NULL', $finalQuery);
                } else {
                    $finalQuery = str_replace($key, $value, $finalQuery);
                }
            }
            $logEntry .= "\n";
        }
        
        $logEntry .= "REQUÊTE SQL FINALE (PRÊTE POUR PHPMYADMIN):\n";
        $logEntry .= str_repeat("-", 50) . "\n";
        $logEntry .= $finalQuery . ";\n";
        $logEntry .= str_repeat("-", 50) . "\n";
        
        if ($error) {
            $logEntry .= "ERREUR:\n";
            $logEntry .= "$error\n";
            $logEntry .= str_repeat("-", 30) . "\n";
        }
        
        $logEntry .= str_repeat("=", 80) . "\n";
        
        // Écrire dans le fichier (avec gestion d'erreur)
        @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    } catch (Exception $e) {
        // Ignorer les erreurs de logging pour ne pas affecter l'API
    }
}

function clearSQLLog() {
    try {
        $logFile = __DIR__ . '/update_sql_debug_log.txt';
        $header = "=== LOG DEBUG SQL - UPDATE DÉCHETS ===\n";
        $header .= "Généré le: " . date('Y-m-d H:i:s') . "\n";
        $header .= "Tables utilisées: releves_dechets + dechets_detail\n\n";
        @file_put_contents($logFile, $header);
    } catch (Exception $e) {
        // Ignorer les erreurs de logging
    }
}

function logReceivedData($data) {
    try {
        $logFile = __DIR__ . '/update_sql_debug_log.txt';
        $timestamp = date('Y-m-d H:i:s');
        
        $logEntry = "\n" . str_repeat("*", 60) . "\n";
        $logEntry .= "DONNÉES REÇUES POUR UPDATE - $timestamp\n";
        $logEntry .= str_repeat("*", 60) . "\n";
        $logEntry .= "JSON BRUT:\n";
        $logEntry .= json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n";
        $logEntry .= str_repeat("*", 60) . "\n";
        
        @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    } catch (Exception $e) {
        // Ignorer les erreurs de logging
    }
}

// ===== CONFIGURATION BASE DE DONNÉES =====
try {
    // Configuration selon votre database.php
    $host = 'localhost';
    $dbname = 'softdgid_onl';
    $username = 'softdgid_pape';
    $password = 'P@sser0101';
    
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    logSQLQuery("-- ERREUR CONNEXION BDD: " . $e->getMessage(), []);
    sendJsonResponse([
        'success' => false,
        'message' => 'Erreur de connexion à la base de données',
        'error' => 'Connexion impossible'
    ], 500);
}

// ===== VALIDATION AUTHENTIFICATION =====
// Mode développement : désactiver temporairement l'authentification
$user = [
    'id' => 1,
    'nom' => 'Admin',
    'prenom' => 'Test',
    'email' => 'admin@test.com',
    'role' => 'admin',
    'nom_complet' => 'Test Admin'
];

// Version avec authentification (à décommenter quand le système token sera prêt)
/*
$token = getAuthToken();
$user = validateToken($token, $pdo);

if (!$user) {
    sendJsonResponse([
        'success' => false,
        'message' => 'Token d\'authentification invalide ou expiré'
    ], 401);
}
*/

// ===== TRAITEMENT PRINCIPAL =====
try {
    // Nettoyer le log au début
    clearSQLLog();
    
    // Récupérer et décoder les données
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    // Logger les données reçues
    logReceivedData($data);
    
    if (!$data) {
        throw new Exception('Données JSON invalides ou vides');
    }
    
    // Vérifier que l'ID est fourni
    if (!isset($data['id']) || empty($data['id'])) {
        throw new Exception('ID du relevé manquant');
    }
    
    $releve_id = (int)$data['id'];
    
    // Vérifier que le relevé existe et appartient à l'utilisateur ou qu'il a les droits
    $sql_check = "SELECT id FROM releves_dechets WHERE id = :id";
    $stmt_check = $pdo->prepare($sql_check);
    $stmt_check->execute([':id' => $releve_id]);
    
    if (!$stmt_check->fetch()) {
        throw new Exception("Relevé avec ID $releve_id non trouvé");
    }
    
    logSQLQuery("-- RELEVÉ EXISTANT CONFIRMÉ: ID = $releve_id", []);
    
    // Commencer une transaction
    $pdo->beginTransaction();
    logSQLQuery("-- DÉBUT TRANSACTION UPDATE", []);
    
    // ===== ÉTAPE 1: MISE À JOUR DU RELEVÉ PRINCIPAL =====
    $sql_update_releve = "UPDATE releves_dechets SET 
        code_referent = :code_referent,
        date_releve = :date_releve,
        heure_debut = :heure_debut,
        latitude_debut = :latitude_debut,
        longitude_debut = :longitude_debut,
        latitude_fin = :latitude_fin,
        longitude_fin = :longitude_fin,
        orientation_plage = :orientation_plage,
        courant_dominant = :courant_dominant,
        vent_dominant = :vent_dominant,
        evenements_extremes = :evenements_extremes,
        autre_evenement = :autre_evenement,
        statut = :statut,
        updated_at = NOW()
    WHERE id = :id";
    
    $params_update_releve = [
        ':id' => $releve_id,
        ':code_referent' => $data['code_referent'] ?? '',
        ':date_releve' => !empty($data['date_releve']) ? date('Y-m-d', strtotime($data['date_releve'])) : date('Y-m-d'),
        ':heure_debut' => $data['heure_debut'] ?? date('H:i:s'),
        ':latitude_debut' => !empty($data['latitude_debut']) ? (float)$data['latitude_debut'] : null,
        ':longitude_debut' => !empty($data['longitude_debut']) ? (float)$data['longitude_debut'] : null,
        ':latitude_fin' => !empty($data['latitude_fin']) ? (float)$data['latitude_fin'] : null,
        ':longitude_fin' => !empty($data['longitude_fin']) ? (float)$data['longitude_fin'] : null,
        ':orientation_plage' => $data['orientation_plage'] ?? null,
        ':courant_dominant' => $data['courant_dominant'] ?? null,
        ':vent_dominant' => $data['vent_dominant'] ?? null,
        ':evenements_extremes' => isset($data['evenements_extremes']) ? json_encode($data['evenements_extremes']) : null,
        ':autre_evenement' => $data['autre_evenement'] ?? null,
        ':statut' => $data['statut'] ?? 'brouillon'
    ];
    
    // Logger et exécuter la mise à jour du relevé
    logSQLQuery($sql_update_releve, $params_update_releve);
    
    try {
        $stmt_update_releve = $pdo->prepare($sql_update_releve);
        $result_update_releve = $stmt_update_releve->execute($params_update_releve);
        
        if (!$result_update_releve) {
            $error = implode(', ', $stmt_update_releve->errorInfo());
            logSQLQuery($sql_update_releve, $params_update_releve, "ERREUR UPDATE RELEVÉ: $error");
            throw new Exception("Erreur lors de la mise à jour du relevé: $error");
        }
        
        logSQLQuery("-- SUCCÈS UPDATE RELEVÉ: ID = $releve_id", []);
        
    } catch (Exception $e) {
        logSQLQuery($sql_update_releve, $params_update_releve, "EXCEPTION UPDATE RELEVÉ: " . $e->getMessage());
        throw $e;
    }
    
    // ===== ÉTAPE 2: GESTION DES DÉCHETS DÉTAILS - VERSION CORRIGÉE =====
    
    // SOLUTION: Sauvegarder les photos existantes avant suppression
    $sql_backup_photos = "SELECT id, categorie, photo FROM dechets_detail WHERE releve_id = :releve_id AND photo IS NOT NULL";
    $stmt_backup = $pdo->prepare($sql_backup_photos);
    $stmt_backup->execute([':releve_id' => $releve_id]);
    $existing_photos = [];
    
    while ($row = $stmt_backup->fetch()) {
        $existing_photos[$row['categorie']] = $row['photo'];
    }
    
    logSQLQuery($sql_backup_photos, [':releve_id' => $releve_id]);
    logSQLQuery("-- PHOTOS SAUVEGARDÉES: " . count($existing_photos) . " catégories", []);
    
    // Supprimer tous les anciens détails
    $sql_delete_details = "DELETE FROM dechets_detail WHERE releve_id = :releve_id";
    $params_delete = [':releve_id' => $releve_id];
    
    logSQLQuery($sql_delete_details, $params_delete);
    
    try {
        $stmt_delete = $pdo->prepare($sql_delete_details);
        $result_delete = $stmt_delete->execute($params_delete);
        
        if (!$result_delete) {
            $error = implode(', ', $stmt_delete->errorInfo());
            logSQLQuery($sql_delete_details, $params_delete, "ERREUR DELETE DÉTAILS: $error");
            throw new Exception("Erreur lors de la suppression des anciens détails: $error");
        }
        
        logSQLQuery("-- SUCCÈS DELETE ANCIENS DÉTAILS", []);
        
    } catch (Exception $e) {
        logSQLQuery($sql_delete_details, $params_delete, "EXCEPTION DELETE DÉTAILS: " . $e->getMessage());
        throw $e;
    }
    
    // Maintenant, insérer les nouveaux détails avec préservation des photos
    $dechets_count = 0;
    $dechets_errors = [];
    $photos_preserved = 0;
    $photos_new = 0;
    
    if (isset($data['dechets']) && is_array($data['dechets']) && !empty($data['dechets'])) {
        logSQLQuery("-- DÉBUT INSERTION NOUVEAUX DÉCHETS DÉTAILS: " . count($data['dechets']) . " éléments", []);
        
        foreach ($data['dechets'] as $index => $dechet) {
            // Valider les données du déchet
            if (!isset($dechet['categorie'])) {
                $error_msg = "Déchet $index: catégorie manquante";
                logSQLQuery("-- ERREUR VALIDATION DÉCHET $index", [], $error_msg);
                $dechets_errors[] = $error_msg;
                continue;
            }
            
            // SOLUTION: Déterminer quelle photo utiliser
            $photo_to_use = null;
            $photo_source = "aucune";
            
            // 1. Si une nouvelle photo est fournie dans les données
            if (isset($dechet['photo']) && is_string($dechet['photo']) && trim($dechet['photo']) !== '') {
                $photo_to_use = trim($dechet['photo']);
                $photo_source = "nouvelle";
                $photos_new++;
                logSQLQuery("-- UTILISATION NOUVELLE PHOTO pour catégorie: " . $dechet['categorie'], []);
            }
            // 2. Sinon, utiliser l'ancienne photo si elle existe
            elseif (isset($existing_photos[$dechet['categorie']])) {
                $photo_to_use = $existing_photos[$dechet['categorie']];
                $photo_source = "préservée";
                $photos_preserved++;
                logSQLQuery("-- RÉUTILISATION ANCIENNE PHOTO pour catégorie: " . $dechet['categorie'], []);
            }
            // 3. Pas de photo du tout
            else {
                logSQLQuery("-- AUCUNE PHOTO pour catégorie: " . $dechet['categorie'], []);
            }
            
            // Préparer la requête d'insertion UNIFIÉE
            $sql_insert_dechet = "INSERT INTO dechets_detail (
                releve_id,
                categorie,
                nombre_articles,
                poids,
                source_pollution,
                photo,
                created_at
            ) VALUES (
                :releve_id,
                :categorie,
                :nombre_articles,
                :poids,
                :source_pollution,
                :photo,
                NOW()
            )";
            
            $params_insert_dechet = [
                ':releve_id' => $releve_id,
                ':categorie' => $dechet['categorie'],
                ':nombre_articles' => (int)($dechet['nombre_articles'] ?? 0),
                ':poids' => (float)($dechet['poids'] ?? 0.0),
                ':source_pollution' => $dechet['source_pollution'] ?? '6',
                ':photo' => $photo_to_use  // Peut être NULL si pas de photo
            ];
            
            // Logger la requête déchet
            logSQLQuery($sql_insert_dechet, $params_insert_dechet);
            
            try {
                $stmt_insert_dechet = $pdo->prepare($sql_insert_dechet);
                $result_insert_dechet = $stmt_insert_dechet->execute($params_insert_dechet);
                
                if (!$result_insert_dechet) {
                    $error = implode(', ', $stmt_insert_dechet->errorInfo());
                    logSQLQuery($sql_insert_dechet, $params_insert_dechet, "ERREUR INSERT DÉCHET $index: $error");
                    $dechets_errors[] = "Déchet $index: $error";
                } else {
                    $dechet_id = $pdo->lastInsertId();
                    logSQLQuery("-- SUCCÈS INSERT DÉCHET $index: ID = $dechet_id (photo: $photo_source)", []);
                    $dechets_count++;
                }
                
            } catch (Exception $e) {
                $error_msg = "EXCEPTION INSERT DÉCHET $index: " . $e->getMessage();
                logSQLQuery($sql_insert_dechet, $params_insert_dechet, $error_msg);
                $dechets_errors[] = $error_msg;
            }
        }
    } else {
        logSQLQuery("-- ATTENTION: Aucun déchet à insérer ou données invalides", []);
    }
    
    // ===== VALIDATION ET COMMIT =====
    if (empty($dechets_errors)) {
        $pdo->commit();
        logSQLQuery("-- COMMIT TRANSACTION UPDATE RÉUSSIE", []);
        
        sendJsonResponse([
            'success' => true,
            'message' => 'Relevé de déchets mis à jour avec succès',
            'data' => [
                'releve_id' => $releve_id,
                'dechets_count' => $dechets_count,
                'photos_preserved' => $photos_preserved,
                'photos_new' => $photos_new,
                'photos_total' => $photos_preserved + $photos_new,
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $user['nom_complet']
            ]
        ]);
        
    } else {
        $pdo->rollback();
        logSQLQuery("-- ROLLBACK TRANSACTION UPDATE", [], "Erreurs déchets: " . implode(', ', $dechets_errors));
        
        sendJsonResponse([
            'success' => false,
            'message' => 'Erreurs lors de la mise à jour des déchets',
            'errors' => $dechets_errors,
            'data' => [
                'releve_id' => $releve_id,
                'dechets_inserted' => $dechets_count,
                'dechets_failed' => count($dechets_errors),
                'photos_preserved' => $photos_preserved,
                'photos_new' => $photos_new
            ]
        ], 400);
    }
    
} catch (Exception $e) {
    // Rollback en cas d'erreur
    if ($pdo->inTransaction()) {
        $pdo->rollback();
        logSQLQuery("-- ROLLBACK TRANSACTION UPDATE (EXCEPTION)", [], $e->getMessage());
    }
    
    logSQLQuery("-- ERREUR GÉNÉRALE UPDATE", [], $e->getMessage());
    
    sendJsonResponse([
        'success' => false,
        'message' => 'Erreur serveur lors de la mise à jour',
        'error' => $e->getMessage(),
        'debug' => [
            'file' => basename(__FILE__),
            'line' => __LINE__
        ]
    ], 500);
}

// S'assurer qu'aucun contenu supplémentaire n'est envoyé
exit();
?>