<?php
// api/export/profils.php - VERSION SANS VÉRIFICATION TOKEN + CONNEXION DIRECTE
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Gestion des requêtes OPTIONS pour CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Activation des erreurs pour debug
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Fonction de réponse JSON sécurisée
function safeJsonResponse($success, $message = '', $data = null, $code = 200) {
    if (ob_get_level()) {
        ob_clean();
    }
    
    http_response_code($code);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

try {
    error_log("🚀 Démarrage export profils - " . date('Y-m-d H:i:s'));
    
    // CONNEXION DATABASE DIRECTE (SANS INCLUDE)
    try {
        $pdo = new PDO(
            "mysql:host=localhost;dbname=softdgid_onl;charset=utf8mb4",
            "softdgid_pape",
            "P@sser0101",
            [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false
            ]
        );
        error_log("✅ Connexion DB réussie");
    } catch (PDOException $e) {
        error_log("❌ Erreur DB: " . $e->getMessage());
        safeJsonResponse(false, 'Erreur de connexion à la base de données', ['error' => $e->getMessage()], 500);
    }
    
    // UTILISATEUR PAR DÉFAUT (SANS AUTHENTIFICATION)
    $currentUser = [
        'nom' => 'Système',
        'prenom' => 'Export',
        'email' => 'system@export.com',
        'role' => 'admin',
        'site_id' => null
    ];
    
    // RÉCUPÉRATION DES PARAMÈTRES
    $format = $_GET['format'] ?? 'json';
    $preview = isset($_GET['preview']) && $_GET['preview'] === 'true';
    $debug = isset($_GET['debug']) && $_GET['debug'] === '1';
    $siteId = $_GET['site_id'] ?? null;
    $limit = $preview ? 10 : null;
    
    error_log("📋 Paramètres: format=$format, preview=" . ($preview ? 'true' : 'false') . ", site_id=$siteId");
    
    // GESTION DU MODE PREVIEW
    if ($preview) {
        safeJsonResponse(true, 'API Export Profils disponible', [
            'user' => $currentUser['nom'] . ' ' . $currentUser['prenom'],
            'timestamp' => date('Y-m-d H:i:s'),
            'available_formats' => ['json', 'csv', 'excel'],
            'status' => 'ready'
        ]);
    }
    
    // CONSTRUCTION DE LA REQUÊTE
    $query = "
        SELECT 
            u.id,
            u.nom,
            u.prenom as prenoms,
            u.telephone,
            u.email,
            u.role,
            CASE 
                WHEN u.actif = 1 THEN 'Actif'
                ELSE 'Inactif'
            END as statut,
            u.created_at,
            u.updated_at,
            COALESCE(s.nom, 'Non assigné') as site_nom,
            u.site_id,
            CASE 
                WHEN u.role = 'admin' THEN 'Administrateur'
                WHEN u.role = 'superviseur' THEN 'Superviseur'
                WHEN u.role = 'controleur' THEN 'Contrôleur'
                WHEN u.role = 'referent' THEN 'Référent'
                ELSE u.role
            END as role_libelle
        FROM utilisateurs u
        LEFT JOIN sites s ON u.site_id = s.id
        WHERE 1=1
    ";
    
    $params = [];
    
    // Filtrage par site si nécessaire
    if ($siteId) {
        $query .= " AND u.site_id = ?";
        $params[] = $siteId;
    }
    
    $query .= " ORDER BY u.created_at DESC";
    
    if ($limit) {
        $query .= " LIMIT ?";
        $params[] = $limit;
    }
    
    // EXÉCUTION DE LA REQUÊTE
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $utilisateurs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("📊 Utilisateurs trouvés: " . count($utilisateurs));
    
    // FORMATAGE DES DONNÉES
    $exportData = [];
    foreach ($utilisateurs as $utilisateur) {
        $exportData[] = [
            'ID' => $utilisateur['id'],
            'Nom' => $utilisateur['nom'] ?? '',
            'Prénoms' => $utilisateur['prenoms'] ?? '',
            'Nom Complet' => trim(($utilisateur['prenoms'] ?? '') . ' ' . ($utilisateur['nom'] ?? '')),
            'Téléphone' => $utilisateur['telephone'] ?? '',
            'Email' => $utilisateur['email'] ?? '',
            'Rôle' => $utilisateur['role_libelle'] ?? '',
            'Statut' => $utilisateur['statut'] ?? '',
            'Site' => $utilisateur['site_nom'] ?? '',
            'Date Création' => $utilisateur['created_at'] ? date('d/m/Y H:i', strtotime($utilisateur['created_at'])) : '',
            'Dernière MAJ' => $utilisateur['updated_at'] ? date('d/m/Y H:i', strtotime($utilisateur['updated_at'])) : 'N/A'
        ];
    }
    
    // STATISTIQUES
    $stats = [
        'total_utilisateurs' => count($exportData),
        'admins' => count(array_filter($utilisateurs, function($u) { return $u['role'] === 'admin'; })),
        'superviseurs' => count(array_filter($utilisateurs, function($u) { return $u['role'] === 'superviseur'; })),
        'controleurs' => count(array_filter($utilisateurs, function($u) { return $u['role'] === 'controleur'; })),
        'referents' => count(array_filter($utilisateurs, function($u) { return $u['role'] === 'referent'; })),
        'date_export' => date('Y-m-d H:i:s'),
        'exported_by' => $currentUser['nom'] . ' ' . $currentUser['prenom'],
        'auth_user' => 'Aucune authentification requise'
    ];
    
    // RÉPONSE SELON LE FORMAT
    switch ($format) {
        case 'excel':
            generateExcelExportProfils($exportData, $stats);
            break;
            
        case 'csv':
            generateCSVExportProfils($exportData);
            break;
            
        case 'pdf':
            safeJsonResponse(true, 'Données prêtes pour export PDF', [
                'data' => $exportData,
                'stats' => $stats,
                'title' => 'Liste des Utilisateurs ONL',
                'subtitle' => 'Export du ' . date('d/m/Y à H:i'),
                'filename' => 'utilisateurs_onl_' . date('Y-m-d_H-i') . '.pdf'
            ]);
            break;
            
        default: // json
            safeJsonResponse(true, $preview ? 'Aperçu des utilisateurs' : 'Export des utilisateurs réussi', [
                'data' => $exportData,
                'stats' => $stats,
                'preview' => $preview,
                'total_records' => count($exportData),
                'debug_info' => $debug ? [
                    'query' => $query,
                    'params' => $params,
                    'auth_user' => 'Pas d\'authentification'
                ] : null
            ]);
    }
    
} catch (Exception $e) {
    error_log("❌ Erreur fatale export: " . $e->getMessage());
    error_log("📍 Fichier: " . $e->getFile() . " Ligne: " . $e->getLine());
    
    safeJsonResponse(false, 'Erreur lors de l\'export: ' . $e->getMessage(), [
        'error_details' => [
            'message' => $e->getMessage(),
            'file' => basename($e->getFile()),
            'line' => $e->getLine()
        ]
    ], 500);
}

/**
 * Génère un export Excel pour les profils utilisateurs
 */
function generateExcelExportProfils($data, $stats) {
    // Vérifier si PhpSpreadsheet est disponible
    $vendorPaths = [
        __DIR__ . '/../vendor/autoload.php',
        __DIR__ . '/../../vendor/autoload.php',
        __DIR__ . '/../../../vendor/autoload.php'
    ];
    
    $vendorFound = false;
    foreach ($vendorPaths as $path) {
        if (file_exists($path)) {
            require_once $path;
            $vendorFound = true;
            break;
        }
    }
    
    if (!$vendorFound) {
        // Fallback vers CSV si PhpSpreadsheet non disponible
        error_log("⚠️ PhpSpreadsheet non trouvé, fallback CSV");
        generateCSVExportProfils($data);
        return;
    }
    
    try {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        
        // Feuille 1: Données principales
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Utilisateurs');
        
        // Titre
        $sheet->setCellValue('A1', 'Liste des Utilisateurs ONL');
        $sheet->mergeCells('A1:K1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);
        
        // En-têtes des colonnes
        $headers = [
            'ID', 'Nom', 'Prénoms', 'Nom Complet', 'Téléphone',
            'Email', 'Rôle', 'Statut', 'Site', 'Date Création', 'Dernière MAJ'
        ];
        
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . '3', $header);
            $col++;
        }
        
        // Style des en-têtes
        $headerRange = 'A3:K3';
        $sheet->getStyle($headerRange)->applyFromArray([
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4']
            ],
            'borders' => [
                'allBorders' => ['borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN]
            ]
        ]);
        
        // Données
        $row = 4;
        foreach ($data as $utilisateur) {
            $sheet->setCellValue('A' . $row, $utilisateur['ID']);
            $sheet->setCellValue('B' . $row, $utilisateur['Nom']);
            $sheet->setCellValue('C' . $row, $utilisateur['Prénoms']);
            $sheet->setCellValue('D' . $row, $utilisateur['Nom Complet']);
            $sheet->setCellValue('E' . $row, $utilisateur['Téléphone']);
            $sheet->setCellValue('F' . $row, $utilisateur['Email']);
            $sheet->setCellValue('G' . $row, $utilisateur['Rôle']);
            $sheet->setCellValue('H' . $row, $utilisateur['Statut']);
            $sheet->setCellValue('I' . $row, $utilisateur['Site']);
            $sheet->setCellValue('J' . $row, $utilisateur['Date Création']);
            $sheet->setCellValue('K' . $row, $utilisateur['Dernière MAJ']);
            $row++;
        }
        
        // Auto-ajuster les colonnes
        foreach (range('A', 'K') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        // Feuille 2: Statistiques
        $statsSheet = $spreadsheet->createSheet();
        $statsSheet->setTitle('Statistiques');
        
        $statsSheet->setCellValue('A1', 'Statistiques des Utilisateurs');
        $statsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        
        $row = 3;
        foreach ($stats as $key => $value) {
            if ($key !== 'date_export' && $key !== 'exported_by' && $key !== 'auth_user') {
                $label = str_replace('_', ' ', ucwords($key, '_'));
                $statsSheet->setCellValue('A' . $row, $label . ':');
                $statsSheet->setCellValue('B' . $row, $value);
                $row++;
            }
        }
        
        // Métadonnées
        $row += 2;
        $statsSheet->setCellValue('A' . $row, 'Date d\'export:');
        $statsSheet->setCellValue('B' . $row, $stats['date_export']);
        $row++;
        $statsSheet->setCellValue('A' . $row, 'Exporté par:');
        $statsSheet->setCellValue('B' . $row, $stats['exported_by']);
        
        // Auto-ajuster les colonnes
        foreach (range('A', 'B') as $col) {
            $statsSheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        // Sélectionner la première feuille
        $spreadsheet->setActiveSheetIndex(0);
        
        // Générer le fichier
        $filename = 'utilisateurs_onl_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save('php://output');
        exit();
        
    } catch (Exception $e) {
        error_log("❌ Erreur Excel: " . $e->getMessage());
        generateCSVExportProfils($data);
    }
}

/**
 * Génère un export CSV pour les profils utilisateurs
 */
function generateCSVExportProfils($data) {
    $filename = 'utilisateurs_onl_' . date('Y-m-d_H-i-s') . '.csv';
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    // Nettoyer le buffer
    if (ob_get_level()) {
        ob_clean();
    }
    
    $output = fopen('php://output', 'w');
    
    // BOM UTF-8 pour Excel
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // En-têtes
    if (!empty($data)) {
        $headers = array_keys($data[0]);
        fputcsv($output, $headers, ';');
        
        // Données
        foreach ($data as $row) {
            fputcsv($output, array_map(function($value) {
                return str_replace('"', '""', $value);
            }, $row), ';');
        }
    }
    
    fclose($output);
    exit();
}
?>