<?php
// =======================================================================
// FICHIER: profils/create.php (ADAPTÉ À VOTRE BASE EXISTANTE)
// =======================================================================

// Inclure la classe Database
require_once '../config/database.php';

// Connexion à la base de données
$database = new Database();
$pdo = $database->getConnection();

if (!$pdo) {
    jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
}

// Lire les données JSON
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    jsonResponse(false, 'Données JSON invalides', null, 400);
}

// =======================================================================
// GESTION DU MODE DEBUG
// =======================================================================

$skipDistanceValidation = isset($input['skip_distance_validation']) && $input['skip_distance_validation'] === true;
$debugMode = isset($input['debug_mode']) && $input['debug_mode'] === true;
$forceSave = isset($input['force_save']) && $input['force_save'] === true;

if ($debugMode || $skipDistanceValidation || $forceSave) {
    error_log("Mode debug profils activé - Flags: skip=" . ($skipDistanceValidation ? 'true' : 'false') . 
             ", debug=" . ($debugMode ? 'true' : 'false') . 
             ", force=" . ($forceSave ? 'true' : 'false'));
}

// =======================================================================
// VALIDATION DES DONNÉES D'ENTRÉE
// =======================================================================

$required_fields = [
    'code_referent', 'numero_profil_borne', 'site_id', 'date_releve', 
    'heure_debut', 'latitude_borne', 'longitude_borne', 'altitude_borne',
    'maree_type', 'mesures_topographiques'
];

foreach ($required_fields as $field) {
    if (!isset($input[$field]) || (is_string($input[$field]) && trim($input[$field]) === '')) {
        jsonResponse(false, "Champ requis manquant: $field", null, 400);
    }
}

// Extraction des données
$code_referent = trim($input['code_referent']);
$numero_profil_borne = trim($input['numero_profil_borne']);
$site_id = (int)$input['site_id'];
$date_releve = $input['date_releve'];
$heure_debut = $input['heure_debut'];
$latitude_borne = (float)$input['latitude_borne'];
$longitude_borne = (float)$input['longitude_borne'];
$altitude_borne = (float)$input['altitude_borne'];
$maree_type = $input['maree_type'];
$evenements_extremes = $input['evenements_extremes'] ?? '';
$autre_evenement = $input['autre_evenement'] ?? '';
$statut = $input['statut'] ?? 'brouillon';
$mesures_topographiques = $input['mesures_topographiques'];

// Déterminer l'utilisateur_id (obligatoire dans votre base)
$utilisateur_id = 3; // BABACAR SAMB par défaut, à adapter selon votre système d'auth

// Validation des coordonnées GPS
if ($latitude_borne < -90 || $latitude_borne > 90 || 
    $longitude_borne < -180 || $longitude_borne > 180) {
    jsonResponse(false, 'Coordonnées GPS de la borne invalides', null, 400);
}

// Validation du type de marée
if (!in_array($maree_type, ['haute', 'basse'])) {
    jsonResponse(false, 'Type de marée invalide. Doit être "haute" ou "basse"', null, 400);
}

// Validation des mesures topographiques
if (!is_array($mesures_topographiques) || empty($mesures_topographiques)) {
    jsonResponse(false, 'Mesures topographiques manquantes ou invalides', null, 400);
}

// =======================================================================
// GESTION DES BORNES DE RÉFÉRENCE (SIMULATION - pas de table dédiée)
// =======================================================================

function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371000; // mètres
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat/2) * sin($dLat/2) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * sin($dLon/2) * sin($dLon/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    return $earthRadius * $c;
}

// Récupérer la dernière borne de référence depuis profils_topographiques existants
$borne_ref_lat = $latitude_borne;
$borne_ref_lon = $longitude_borne;
$is_new_reference = true;

try {
    // Chercher un profil existant pour ce site comme référence
    $stmt = $pdo->prepare("
        SELECT latitude_borne, longitude_borne 
        FROM profils_topographiques 
        WHERE site_id = ? 
        AND latitude_borne IS NOT NULL 
        AND longitude_borne IS NOT NULL
        ORDER BY created_at ASC 
        LIMIT 1
    ");
    $stmt->execute([$site_id]);
    $existing_profile = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existing_profile) {
        $borne_ref_lat = (float)$existing_profile['latitude_borne'];
        $borne_ref_lon = (float)$existing_profile['longitude_borne'];
        $is_new_reference = false;
        error_log("Borne de référence trouvée depuis profil existant pour site $site_id: $borne_ref_lat, $borne_ref_lon");
    } else {
        error_log("Première mesure pour site $site_id - nouvelle borne de référence: $borne_ref_lat, $borne_ref_lon");
    }
} catch(PDOException $e) {
    error_log("Erreur récupération borne référence profils: " . $e->getMessage());
}

// =======================================================================
// VALIDATION DE DISTANCE
// =======================================================================

if (!$skipDistanceValidation && !$debugMode && !$forceSave && !$is_new_reference) {
    $distance_to_reference = calculateDistance($latitude_borne, $longitude_borne, $borne_ref_lat, $borne_ref_lon);
    
    if ($distance_to_reference > 20) {
        jsonResponse(false, 'Position borne: Distance trop élevée: ' . round($distance_to_reference, 2) . 'm (max: 20m)', [
            'debug_info' => [
                'distance_calculated' => round($distance_to_reference, 2),
                'max_allowed' => 20
            ]
        ], 400);
    }
    
    error_log("Validation de distance profil OK - Distance à référence: " . round($distance_to_reference, 2) . "m");
} else {
    if (!$is_new_reference) {
        $distance_to_reference = calculateDistance($latitude_borne, $longitude_borne, $borne_ref_lat, $borne_ref_lon);
        error_log("MODE DEBUG PROFILS: Validation de distance ignorée - Distance: " . round($distance_to_reference, 2) . "m");
    } else {
        error_log("Nouvelle borne de référence profils - pas de validation de distance");
        $distance_to_reference = 0;
    }
}

// =======================================================================
// INSERTION EN BASE DE DONNÉES
// =======================================================================

try {
    $pdo->beginTransaction();
    
    // Insertion du profil principal (AVEC utilisateur_id requis)
    $stmt = $pdo->prepare("
        INSERT INTO profils_topographiques (
            code_referent, numero_profil_borne, utilisateur_id, site_id, 
            date_releve, heure_debut, latitude_borne, longitude_borne, altitude_borne,
            maree_type, evenements_extremes, autre_evenement, statut,
            created_at
        ) VALUES (
            ?, ?, ?, ?, ?,
            ?, ?, ?, ?,
            ?, ?, ?, ?,
            NOW()
        )
    ");
    
    $stmt->execute([
        $code_referent, $numero_profil_borne, $utilisateur_id, $site_id,
        $date_releve, $heure_debut, $latitude_borne, $longitude_borne, $altitude_borne,
        $maree_type, $evenements_extremes, $autre_evenement, $statut
    ]);
    
    $profil_id = $pdo->lastInsertId();
    
    // Insertion des mesures topographiques
    $stmt_mesure = $pdo->prepare("
        INSERT INTO mesures_topographiques (
            profil_id, numero_ordre, distance_metres, denivele_cm,
            denivele_cumule_cm, altitude_metres, commentaires,
            created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    
    $mesures_count = 0;
    
    foreach ($mesures_topographiques as $mesure) {
        $numero_ordre = (int)($mesure['numero_ordre'] ?? 0);
        $distance_metres = (float)($mesure['distance_metres'] ?? 0.0);
        $denivele_cm = (float)($mesure['denivele_cm'] ?? 0.0);
        $denivele_cumule_cm = (float)($mesure['denivele_cumule_cm'] ?? 0.0);
        $altitude_metres = (float)($mesure['altitude_metres'] ?? 0.0);
        $commentaires = $mesure['commentaires'] ?? '';
        
        if ($numero_ordre > 0) {
            $stmt_mesure->execute([
                $profil_id, $numero_ordre, $distance_metres, $denivele_cm,
                $denivele_cumule_cm, $altitude_metres, $commentaires
            ]);
            
            $mesures_count++;
        }
    }
    
    $pdo->commit();
    
    // Réponse de succès
    jsonResponse(true, 'Profil topographique créé avec succès', [
        'profil_id' => $profil_id,
        'id' => $profil_id,
        'numero_profil_borne' => $numero_profil_borne,
        'mesures_count' => $mesures_count,
        'utilisateur_id' => $utilisateur_id,
        'site_id' => $site_id,
        'date_releve' => $date_releve,
        'heure_debut' => $heure_debut,
        'debug_mode' => $debugMode || $skipDistanceValidation || $forceSave,
        'borne_reference' => [
            'latitude' => $borne_ref_lat,
            'longitude' => $borne_ref_lon,
            'is_new' => $is_new_reference,
            'distance_from_borne' => $distance_to_reference ?? 0
        ]
    ]);
    
    error_log("✅ Profil topographique créé avec succès: ID=$profil_id, Mesures=$mesures_count, Utilisateur=$utilisateur_id");
    
} catch(PDOException $e) {
    $pdo->rollback();
    
    error_log("❌ Erreur création profil topographique: " . $e->getMessage());
    
    jsonResponse(false, 'Erreur lors de la création du profil topographique', [
        'error' => $e->getMessage(),
        'debug_info' => [
            'timestamp' => date('Y-m-d H:i:s'),
            'sql_error' => $e->getCode()
        ]
    ], 500);
}

?>