<?php
/**
 * API pour récupérer un profil topographique détaillé
 * Emplacement : /public_html/onl/api/profils/get.php
 */

// Inclusion du fichier de configuration (qui contient déjà jsonResponse)
require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    // Vérifier que l'ID est fourni
    $profil_id = isset($_GET['id']) ? intval($_GET['id']) : null;
    
    if (!$profil_id) {
        jsonResponse(false, 'ID du profil requis', null, 400);
    }
    
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Requête pour récupérer le profil avec toutes ses relations
    // CORRIGÉE selon la vraie structure de votre base
    $sql = "SELECT 
                p.id,
                p.code_referent,
                p.numero_profil_borne,
                p.date_releve,
                p.heure_debut,
                p.latitude_borne,
                p.longitude_borne,
                p.altitude_borne,
                p.azimut,
                p.maree_type,
                p.maree,
                p.evenements_extremes,
                p.autre_evenement,
                p.statut,
                p.created_at,
                p.utilisateur_id,
                p.site_id,
                u.nom as referent_nom,
                u.prenom as referent_prenom,
                u.email as referent_email,
                u.telephone as referent_telephone,
                s.nom_site,
                s.region,
                s.latitude as site_latitude,
                s.longitude as site_longitude
            FROM profils_topographiques p
            LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id
            LEFT JOIN sites s ON p.site_id = s.id
            WHERE p.id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$profil_id]);
    $profil = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$profil) {
        jsonResponse(false, 'Profil topographique non trouvé', null, 404);
    }
    
    // Récupérer les mesures topographiques associées
    // NOMS DE COLONNES CORRIGES selon votre vraie structure
    $mesuresSql = "SELECT 
                        id,
                        numero_ordre,
                        distance_metres,
                        denivele_cm,
                        denivele_cumule_cm,
                        altitude_metres,
                        commentaires,
                        created_at
                    FROM mesures_topographiques 
                    WHERE profil_id = ? 
                    ORDER BY numero_ordre ASC";
    
    $mesuresStmt = $pdo->prepare($mesuresSql);
    $mesuresStmt->execute([$profil_id]);
    $mesures = $mesuresStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Formater les mesures
    $mesuresFormatees = [];
    $denivele_cumule = 0;
    
    foreach ($mesures as $mesure) {
        $denivele_cumule += floatval($mesure['denivele_cm'] ?? 0);
        
        $mesuresFormatees[] = [
            'id' => intval($mesure['id']),
            'numero_ordre' => intval($mesure['numero_ordre']),
            'distance' => floatval($mesure['distance_metres'] ?? 0),
            'denivele' => floatval($mesure['denivele_cm'] ?? 0),
            'denivele_cumule' => floatval($mesure['denivele_cumule_cm'] ?? $denivele_cumule),
            'altitude' => floatval($mesure['altitude_metres'] ?? 0),
            'commentaires' => $mesure['commentaires'],
            'created_at' => $mesure['created_at']
        ];
    }
    
    // Parser les événements extrêmes (stockés en JSON ou texte séparé par virgule)
    $evenements = [];
    if (!empty($profil['evenements_extremes'])) {
        $evenementsData = $profil['evenements_extremes'];
        
        // Essayer de décoder comme JSON d'abord
        $evenementsJson = json_decode($evenementsData, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($evenementsJson)) {
            $evenements = $evenementsJson;
        } else {
            // Sinon, traiter comme texte séparé par virgule
            $evenements = array_map('trim', explode(',', $evenementsData));
            // Filtrer les éléments vides
            $evenements = array_filter($evenements, function($item) {
                return !empty(trim($item));
            });
        }
    }
    
    // Calculer les statistiques des mesures
    $stats_mesures = [
        'nombre_mesures' => count($mesuresFormatees),
        'distance_totale' => count($mesuresFormatees) > 0 ? array_sum(array_column($mesuresFormatees, 'distance')) : 0,
        'denivele_total' => count($mesuresFormatees) > 0 ? array_sum(array_column($mesuresFormatees, 'denivele')) : 0,
        'altitude_min' => !empty($mesuresFormatees) ? min(array_column($mesuresFormatees, 'altitude')) : 0,
        'altitude_max' => !empty($mesuresFormatees) ? max(array_column($mesuresFormatees, 'altitude')) : 0,
    ];
    
    // Formater la réponse finale
    $profilDetaille = [
        // Informations identification
        'identification' => [
            'id' => intval($profil['id']),
            'code_referent' => $profil['code_referent'],
            'numero_profil_borne' => $profil['numero_profil_borne'],
            'date_releve' => $profil['date_releve'],
            'heure_debut' => $profil['heure_debut'],
            'statut' => $profil['statut'],
            'created_at' => $profil['created_at'],
            'updated_at' => $profil['created_at'] // Pas d'updated_at dans votre table
        ],
        
        // Informations référent
        'referent' => [
            'id' => intval($profil['utilisateur_id'] ?? 0),
            'nom_complet' => trim(($profil['referent_prenom'] ?? '') . ' ' . ($profil['referent_nom'] ?? '')),
            'nom' => $profil['referent_nom'],
            'prenom' => $profil['referent_prenom'],
            'email' => $profil['referent_email'],
            'telephone' => $profil['referent_telephone']
        ],
        
        // Informations site
        'site' => [
            'id' => intval($profil['site_id'] ?? 0),
            'nom_site' => $profil['nom_site'],
            'region' => $profil['region'],
            'latitude' => floatval($profil['site_latitude'] ?? 0),
            'longitude' => floatval($profil['site_longitude'] ?? 0)
        ],
        
        // Position GPS de la borne
        'position_gps' => [
            'latitude' => floatval($profil['latitude_borne'] ?? 0),
            'longitude' => floatval($profil['longitude_borne'] ?? 0),
            'altitude' => floatval($profil['altitude_borne'] ?? 0),
            'azimut' => intval($profil['azimut'] ?? 0)
        ],
        
        // Conditions de marée
        'maree' => [
            'type' => $profil['maree_type'], // 'haute' ou 'basse'
            'details' => $profil['maree'] ? 'Marée ' . $profil['maree_type'] : null
        ],
        
        // Événements extrêmes
        'evenements_extremes' => [
            'liste' => array_values($evenements), // Réindexer le tableau
            'autre_evenement' => $profil['autre_evenement'],
            'nombre_evenements' => count($evenements)
        ],
        
        // Mesures topographiques
        'mesures_topographiques' => [
            'mesures' => $mesuresFormatees,
            'statistiques' => $stats_mesures
        ],
        
        // Métadonnées
        'meta' => [
            'permissions' => [
                'can_view' => true,
                'can_edit' => true,
                'can_validate' => true,
                'can_delete' => true
            ],
            'export_formats' => ['PDF', 'Excel', 'CSV'],
            'last_modified' => $profil['created_at'], // Utiliser created_at car pas d'updated_at
            'database_info' => [
                'server' => $_SERVER['HTTP_HOST'] ?? 'localhost',
                'database' => 'softdgid_toto',
                'table_structure' => 'real_structure_v1'
            ]
        ]
    ];
    
    jsonResponse(true, 'Profil topographique récupéré avec succès', $profilDetaille);
    
} catch (PDOException $e) {
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), [
        'error_type' => 'PDO',
        'error_code' => $e->getCode(),
        'sql_error' => true
    ], 500);
} catch (Exception $e) {
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), [
        'error_type' => 'General'
    ], 500);
}
?>