<?php
/**
 * API pour lister les profils topographiques
 * Emplacement : /public_html/onl/api/profils/list.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

try {
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Requête pour récupérer les profils topographiques
    $sql = "SELECT 
                p.id,
                p.code_referent,
                p.numero_profil_borne,
                p.date_releve,
                p.heure_debut,
                p.latitude_borne,
                p.longitude_borne,
                p.altitude_borne,
                p.azimut,
                p.maree_type,
                p.evenements_extremes,
                p.autre_evenement,
                p.statut,
                p.created_at,
                p.utilisateur_id,
                p.site_id,
                u.nom as referent_nom,
                u.prenom as referent_prenom,
                s.nom_site,
                s.region
            FROM profils_topographiques p
            LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id
            LEFT JOIN sites s ON p.site_id = s.id
            ORDER BY p.created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $profils = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Log pour debug
    error_log("Profils récupérés depuis database.php: " . count($profils));
    
    // Formater les résultats
    $results = [];
    foreach ($profils as $profil) {
        $results[] = [
            'id' => intval($profil['id']),
            'code_referent' => $profil['code_referent'],
            'numero_profil_borne' => $profil['numero_profil_borne'],
            'date_releve' => $profil['date_releve'],
            'heure_debut' => $profil['heure_debut'],
            'latitude' => floatval($profil['latitude_borne'] ?? 0),
            'longitude' => floatval($profil['longitude_borne'] ?? 0),
            'altitude' => floatval($profil['altitude_borne'] ?? 0),
            'azimut' => intval($profil['azimut'] ?? 0),
            'maree_type' => $profil['maree_type'],
            'evenements_extremes' => $profil['evenements_extremes'],
            'autre_evenement' => $profil['autre_evenement'],
            'statut' => $profil['statut'],
            'referent_nom' => trim(($profil['referent_prenom'] ?? '') . ' ' . ($profil['referent_nom'] ?? '')),
            'nom_site' => $profil['nom_site'],
            'region' => $profil['region'],
            'created_at' => $profil['created_at'],
            'utilisateur_id' => intval($profil['utilisateur_id'] ?? 0),
            'site_id' => intval($profil['site_id'] ?? 0),
            'permissions' => [
                'can_view' => true,
                'can_edit' => true,
                'can_validate' => true,
                'can_delete' => true
            ]
        ];
    }
    
    // Réponse finale
    $response_data = [
        'profils' => $results,
        'total' => count($results),
        'database_info' => [
            'connection_method' => 'database.php',
            'server' => $_SERVER['HTTP_HOST'] ?? 'localhost',
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    
    jsonResponse(true, count($results) . ' profil(s) trouvé(s)', $response_data);
    
} catch (PDOException $e) {
    error_log("Erreur PDO dans list.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("Erreur générale dans list.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}
?>