<?php
// api/profils/list.php - Liste des profils avec gestion des rôles
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Gestion des requêtes OPTIONS pour CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once '../config/database.php';
require_once '../config/auth_middleware.php';

// Variable globale pour stocker toutes les requêtes SQL
$sqlLog = [];

/**
 * Fonction pour ajouter une requête SQL au log
 */
function logSQL($description, $sql, $params = [], $result = null) {
    global $sqlLog;
    $sqlLog[] = [
        'timestamp' => date('Y-m-d H:i:s'),
        'description' => $description,
        'sql' => $sql,
        'params' => $params,
        'result' => $result
    ];
}

/**
 * Fonction pour sauvegarder toutes les requêtes dans un fichier
 */
function saveSQLLogToFile() {
    global $sqlLog;
    
    $filename = __DIR__ . '/sql_queries_' . date('Y-m-d_H-i-s') . '.txt';
    $content = "================================================================\n";
    $content .= "FICHIER GENERE AUTOMATIQUEMENT PAR api/profils/list.php\n";
    $content .= "Date d'exécution: " . date('Y-m-d H:i:s') . "\n";
    $content .= "URL: " . ($_SERVER['REQUEST_URI'] ?? 'N/A') . "\n";
    $content .= "Méthode: " . ($_SERVER['REQUEST_METHOD'] ?? 'N/A') . "\n";
    $content .= "User Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'N/A') . "\n";
    $content .= "================================================================\n\n";
    
    foreach ($sqlLog as $index => $entry) {
        $content .= "-- ------------------------------------------------\n";
        $content .= "-- REQUETE " . ($index + 1) . ": " . $entry['description'] . "\n";
        $content .= "-- Timestamp: " . $entry['timestamp'] . "\n";
        $content .= "-- ------------------------------------------------\n\n";
        
        // Afficher les paramètres s'il y en a
        if (!empty($entry['params'])) {
            $content .= "-- Paramètres:\n";
            foreach ($entry['params'] as $key => $param) {
                $content .= "-- Param " . ($key + 1) . ": " . var_export($param, true) . "\n";
            }
            $content .= "\n";
        }
        
        // La requête SQL formatée
        $content .= $entry['sql'] . ";\n\n";
        
        // Version avec paramètres remplacés pour test direct dans phpMyAdmin
        if (!empty($entry['params'])) {
            $testSQL = $entry['sql'];
            foreach ($entry['params'] as $param) {
                $pos = strpos($testSQL, '?');
                if ($pos !== false) {
                    if (is_null($param)) {
                        $value = 'NULL';
                    } elseif (is_string($param)) {
                        $value = "'" . addslashes($param) . "'";
                    } elseif (is_bool($param)) {
                        $value = $param ? '1' : '0';
                    } else {
                        $value = $param;
                    }
                    $testSQL = substr_replace($testSQL, $value, $pos, 1);
                }
            }
            $content .= "-- VERSION POUR COPIER-COLLER DANS PHPMYADMIN:\n";
            $content .= $testSQL . ";\n\n";
        }
        
        // Résultat si disponible
        if ($entry['result'] !== null) {
            $content .= "-- Résultat: " . $entry['result'] . "\n";
        }
        
        $content .= "\n";
    }
    
    // Ajouter des requêtes de diagnostic
    $content .= "================================================================\n";
    $content .= "REQUETES DE DIAGNOSTIC SUPPLEMENTAIRES\n";
    $content .= "================================================================\n\n";
    
    $content .= "-- Vérifier la structure des tables\n";
    $content .= "DESCRIBE profils_topographiques;\n";
    $content .= "DESCRIBE utilisateurs;\n";
    $content .= "DESCRIBE sites;\n";
    $content .= "DESCRIBE mesures_topographiques;\n\n";
    
    $content .= "-- Compter les enregistrements\n";
    $content .= "SELECT COUNT(*) as total_profils FROM profils_topographiques;\n";
    $content .= "SELECT COUNT(*) as total_utilisateurs FROM utilisateurs;\n";
    $content .= "SELECT COUNT(*) as total_sites FROM sites;\n";
    $content .= "SELECT COUNT(*) as total_mesures FROM mesures_topographiques;\n\n";
    
    $content .= "-- Échantillon des données\n";
    $content .= "SELECT * FROM profils_topographiques LIMIT 3;\n";
    $content .= "SELECT * FROM utilisateurs LIMIT 3;\n";
    $content .= "SELECT * FROM sites LIMIT 3;\n\n";
    
    $content .= "-- Vérifier les relations\n";
    $content .= "SELECT \n";
    $content .= "    p.id,\n";
    $content .= "    p.utilisateur_id,\n";
    $content .= "    p.site_id,\n";
    $content .= "    u.nom as utilisateur_nom,\n";
    $content .= "    s.nom_site\n";
    $content .= "FROM profils_topographiques p\n";
    $content .= "LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id\n";
    $content .= "LEFT JOIN sites s ON p.site_id = s.id\n";
    $content .= "LIMIT 5;\n\n";
    
    file_put_contents($filename, $content);
    error_log("Fichier SQL généré: " . $filename);
    return $filename;
}

/**
 * Fonction pour retourner une réponse JSON standardisée
 */
function jsonResponse($success, $message, $data = null, $httpCode = 200) {
    // Sauvegarder le log SQL avant de terminer
    saveSQLLogToFile();
    
    http_response_code($httpCode);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    exit;
}

/**
 * Fonction pour obtenir les filtres de données selon le rôle
 */
function getDataFilters($user) {
    $filters = [];
    
    switch ($user['role']) {
        case 'referent':
            // Un référent ne voit que ses propres données
            $filters['user_id'] = $user['id'];
            $filters['site_id'] = $user['site_id'];
            break;
            
        case 'controleur':
            // Un contrôleur voit toutes les données de son site
            $filters['site_id'] = $user['site_id'];
            break;
            
        case 'superviseur':
        case 'admin':
            // Pas de restriction
            break;
    }
    
    logSQL("Filtres appliqués selon le rôle " . $user['role'], "-- Filtres: " . json_encode($filters));
    return $filters;
}

/**
 * Fonction pour vérifier si un utilisateur peut accéder à un site
 */
function canAccessSite($user, $site_id) {
    switch ($user['role']) {
        case 'referent':
        case 'controleur':
            return $user['site_id'] == $site_id;
        case 'superviseur':
        case 'admin':
            return true;
        default:
            return false;
    }
}

/**
 * Fonction pour obtenir les utilisateurs par rôle et site
 */
function getUsersByRoleAndSite($conn, $role, $site_id) {
    try {
        $sql = "SELECT id, nom, prenom FROM utilisateurs WHERE role = ? AND site_id = ? AND status = 'active'";
        logSQL("Récupération des utilisateurs par rôle et site", $sql, [$role, $site_id]);
        
        $stmt = $conn->prepare($sql);
        $stmt->execute([$role, $site_id]);
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        logSQL("Résultat utilisateurs", "-- Nombre d'utilisateurs trouvés: " . count($result));
        return $result;
    } catch (Exception $e) {
        logSQL("Erreur getUsersByRoleAndSite", "-- ERREUR: " . $e->getMessage());
        return [];
    }
}

try {
    logSQL("Début de l'exécution", "-- Démarrage de api/profils/list.php");
    
    // Authentification requise
    $user = requireAuth();
    
    // Log de debug
    error_log("User authenticated: " . json_encode($user));
    logSQL("Authentification réussie", "-- Utilisateur: " . json_encode($user));
    
    // Initialiser la connexion DB
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        logSQL("Erreur connexion DB", "-- Impossible de se connecter à la base de données");
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    logSQL("Connexion DB réussie", "-- Connexion à la base de données établie");
    
    // Récupérer les paramètres
    $statut = isset($_GET['statut']) ? $_GET['statut'] : '';
    $site_id = isset($_GET['site_id']) ? intval($_GET['site_id']) : null;
    $referent_id = isset($_GET['referent_id']) ? intval($_GET['referent_id']) : null;
    
    // Log des paramètres
    error_log("Paramètres reçus - statut: $statut, site_id: $site_id, referent_id: $referent_id");
    logSQL("Paramètres de la requête", "-- statut: '$statut', site_id: $site_id, referent_id: $referent_id");
    
    // Appliquer les filtres selon le rôle
    $filters = getDataFilters($user);
    error_log("Filtres appliqués: " . json_encode($filters));
    
    // Construction de la requête
    $sql = "SELECT 
                p.id,
                p.code_referent,
                p.numero_profil_borne,
                p.date_releve,
                p.heure_debut,
                p.latitude_borne,
                p.longitude_borne,
                p.altitude_borne,
                p.azimut,
                p.maree_type,
                p.evenements_extremes,
                p.autre_evenement,
                p.statut,
                p.created_at,
                p.utilisateur_id,
                p.site_id,
                u.nom as referent_nom,
                u.prenom as referent_prenom,
                s.nom_site,
                s.region,
                COUNT(m.id) as nb_mesures
            FROM profils_topographiques p
            LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id
            LEFT JOIN sites s ON p.site_id = s.id
            LEFT JOIN mesures_topographiques m ON p.id = m.profil_id
            WHERE 1=1";
    
    $params = [];
    
    // Appliquer les filtres de rôle
    if (isset($filters['site_id'])) {
        $sql .= " AND p.site_id = ?";
        $params[] = $filters['site_id'];
    }
    
    if (isset($filters['user_id'])) {
        $sql .= " AND p.utilisateur_id = ?";
        $params[] = $filters['user_id'];
    }
    
    // Filtrer par statut si spécifié
    if (!empty($statut)) {
        $sql .= " AND p.statut = ?";
        $params[] = $statut;
    }
    
    // Filtrer par site si spécifié (et autorisé)
    if ($site_id && canAccessSite($user, $site_id)) {
        $sql .= " AND p.site_id = ?";
        $params[] = $site_id;
    }
    
    // Filtrer par référent si spécifié (pour les contrôleurs)
    if ($referent_id && $user['role'] === 'controleur') {
        $sql .= " AND p.utilisateur_id = ?";
        $params[] = $referent_id;
    }
    
    $sql .= " GROUP BY p.id ORDER BY p.created_at DESC LIMIT 100";
    
    error_log("SQL Query: " . $sql);
    error_log("Parameters: " . json_encode($params));
    
    // Log de la requête principale
    logSQL("Requête principale des profils topographiques", $sql, $params);
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $profils = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("Profils trouvés: " . count($profils));
    logSQL("Résultat de la requête principale", "-- " . count($profils) . " profil(s) trouvé(s)");
    
    // Formater les données
    $results = [];
    foreach ($profils as $profil) {
        $result = [
            'id' => intval($profil['id']),
            'code_referent' => $profil['code_referent'],
            'numero_profil_borne' => $profil['numero_profil_borne'],
            'date_releve' => $profil['date_releve'],
            'heure_debut' => $profil['heure_debut'],
            'latitude' => floatval($profil['latitude_borne']),
            'longitude' => floatval($profil['longitude_borne']),
            'altitude' => floatval($profil['altitude_borne']),
            'azimut' => intval($profil['azimut']),
            'maree_type' => $profil['maree_type'],
            'evenements_extremes' => $profil['evenements_extremes'],
            'autre_evenement' => $profil['autre_evenement'],
            'statut' => $profil['statut'],
            'referent_nom' => trim(($profil['referent_prenom'] ?? '') . ' ' . ($profil['referent_nom'] ?? '')),
            'nom_site' => $profil['nom_site'],
            'region' => $profil['region'],
            'nb_mesures' => intval($profil['nb_mesures']),
            'created_at' => $profil['created_at'],
            'utilisateur_id' => intval($profil['utilisateur_id']),
            'site_id' => intval($profil['site_id'])
        ];
        
        // Ajouter les permissions pour chaque élément
        $result['permissions'] = [
            'can_view' => true, // Si on le voit, on peut le consulter
            'can_edit' => canModifyData($user, $profil['site_id'], $profil['utilisateur_id']),
            'can_validate' => canValidateData($user, $profil['site_id']),
            'can_delete' => canModifyData($user, $profil['site_id'], $profil['utilisateur_id'])
        ];
        
        $results[] = $result;
    }
    
    // Si c'est un contrôleur, ajouter la liste des référents de son site
    $referents = [];
    if ($user['role'] === 'controleur') {
        $referents = getUsersByRoleAndSite($conn, 'referent', $user['site_id']);
    }
    
    $response_data = [
        'profils' => $results,
        'total' => count($results),
        'user_role' => $user['role'],
        'user_site_id' => $user['site_id'],
        'referents' => $referents // Liste des référents pour le contrôleur
    ];
    
    logSQL("Préparation de la réponse finale", "-- " . count($results) . " profil(s) formaté(s) pour la réponse");
    
    jsonResponse(true, count($results) . ' profil(s) trouvé(s)', $response_data);
    
} catch (Exception $e) {
    error_log("Erreur profils/list.php: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    
    logSQL("ERREUR CRITIQUE", "-- ERREUR: " . $e->getMessage() . "\n-- Stack trace: " . $e->getTraceAsString());
    
    jsonResponse(false, 'Erreur lors de la récupération des profils: ' . $e->getMessage(), null, 500);
}

/**
 * Fonction pour vérifier si un utilisateur peut modifier des données
 */
function canModifyData($user, $site_id, $owner_id) {
    switch ($user['role']) {
        case 'referent':
            // Un référent peut modifier ses propres données
            return $user['id'] == $owner_id;
        case 'controleur':
            // Un contrôleur peut modifier les données de son site
            return $user['site_id'] == $site_id;
        case 'superviseur':
        case 'admin':
            // Peut tout modifier
            return true;
        default:
            return false;
    }
}

/**
 * Fonction pour vérifier si un utilisateur peut valider des données
 */
function canValidateData($user, $site_id) {
    switch ($user['role']) {
        case 'controleur':
            // Un contrôleur peut valider les données de son site
            return $user['site_id'] == $site_id;
        case 'superviseur':
        case 'admin':
            // Peut tout valider
            return true;
        default:
            return false;
    }
}
?>