<?php
/**
 * API pour récupérer toutes les mesures topographiques
 * Emplacement : /public_html/onl/api/profils/mesures.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

/**
 * Fonction pour envoyer une réponse JSON standardisée
 */
function jsonResponse($success, $message, $data = null, $status = 200) {
    http_response_code($status);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

try {
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Paramètres optionnels
    $profil_id = isset($_GET['profil_id']) ? intval($_GET['profil_id']) : null;
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : null;
    $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
    
    // Construction de la requête SQL
    $sql = "SELECT 
                m.id,
                m.profil_id,
                m.numero_ordre,
                m.distance_metres,
                m.denivele_cm,
                m.denivele_cumule_cm,
                m.altitude_metres,
                m.commentaires,
                m.created_at,
                p.code_referent,
                p.numero_profil_borne,
                p.date_releve,
                s.nom_site,
                s.region,
                u.nom as referent_nom,
                u.prenom as referent_prenom
            FROM mesures_topographiques m
            LEFT JOIN profils_topographiques p ON m.profil_id = p.id
            LEFT JOIN sites s ON p.site_id = s.id
            LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id";
    
    $conditions = [];
    $params = [];
    
    // Filtrage par profil si spécifié
    if ($profil_id) {
        $conditions[] = "m.profil_id = :profil_id";
        $params[':profil_id'] = $profil_id;
    }
    
    // Ajouter les conditions WHERE si nécessaire
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    // Ordonner par profil puis par numéro d'ordre
    $sql .= " ORDER BY m.profil_id ASC, m.numero_ordre ASC";
    
    // Ajouter la pagination si spécifiée
    if ($limit) {
        $sql .= " LIMIT :limit";
        if ($offset > 0) {
            $sql .= " OFFSET :offset";
        }
        $params[':limit'] = $limit;
        if ($offset > 0) {
            $params[':offset'] = $offset;
        }
    }
    
    // Exécuter la requête
    $stmt = $pdo->prepare($sql);
    
    // Bind des paramètres
    foreach ($params as $key => $value) {
        if ($key === ':limit' || $key === ':offset' || $key === ':profil_id') {
            $stmt->bindValue($key, $value, PDO::PARAM_INT);
        } else {
            $stmt->bindValue($key, $value, PDO::PARAM_STR);
        }
    }
    
    $stmt->execute();
    $mesures = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Log pour debug
    error_log("Mesures topographiques récupérées: " . count($mesures));
    
    // Formater les résultats
    $results = [];
    foreach ($mesures as $mesure) {
        $results[] = [
            'id' => intval($mesure['id']),
            'profil_id' => intval($mesure['profil_id']),
            'numero_ordre' => intval($mesure['numero_ordre']),
            'distance_metres' => floatval($mesure['distance_metres'] ?? 0),
            'denivele_cm' => floatval($mesure['denivele_cm'] ?? 0),
            'denivele_cumule_cm' => floatval($mesure['denivele_cumule_cm'] ?? 0),
            'altitude_metres' => floatval($mesure['altitude_metres'] ?? 0),
            'commentaires' => $mesure['commentaires'] ?? '',
            'created_at' => $mesure['created_at'],
            
            // Informations du profil associé
            'profil_info' => [
                'code_referent' => $mesure['code_referent'],
                'numero_profil_borne' => $mesure['numero_profil_borne'],
                'date_releve' => $mesure['date_releve'],
                'referent_nom' => trim(($mesure['referent_prenom'] ?? '') . ' ' . ($mesure['referent_nom'] ?? '')),
                'site_nom' => $mesure['nom_site'],
                'region' => $mesure['region']
            ]
        ];
    }
    
    // Statistiques supplémentaires
    $stats = [
        'total_mesures' => count($results),
        'profils_avec_mesures' => count(array_unique(array_column($results, 'profil_id'))),
        'derniere_mise_a_jour' => date('Y-m-d H:i:s')
    ];
    
    if ($profil_id) {
        $stats['profil_filtre'] = $profil_id;
    }
    
    // Réponse finale
    $response_data = [
        'mesures' => $results,
        'statistiques' => $stats,
        'parametres' => [
            'profil_id' => $profil_id,
            'limit' => $limit,
            'offset' => $offset
        ]
    ];
    
    jsonResponse(true, count($results) . ' mesure(s) topographique(s) trouvée(s)', $response_data);
    
} catch (PDOException $e) {
    error_log("Erreur PDO dans mesures.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("Erreur générale dans mesures.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}
?>