<?php
/**
 * API pour modifier un profil topographique
 * Emplacement : /public_html/onl/api/profils/update.php
 */

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    // Vérifier la méthode HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'PUT' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
        jsonResponse(false, 'Méthode non autorisée. Utilisez PUT ou POST.', null, 405);
    }
    
    // Récupérer les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        jsonResponse(false, 'Données JSON invalides', null, 400);
    }
    
    // Vérifier l'ID du profil
    $profil_id = isset($data['id']) ? intval($data['id']) : null;
    if (!$profil_id) {
        jsonResponse(false, 'ID du profil requis', null, 400);
    }
    
    // Connexion à la base
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Vérifier que le profil existe
    $checkSql = "SELECT id, statut FROM profils_topographiques WHERE id = ?";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$profil_id]);
    $existingProfil = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$existingProfil) {
        jsonResponse(false, 'Profil non trouvé', null, 404);
    }
    
    // Commencer une transaction
    $pdo->beginTransaction();
    
    try {
        // Mettre à jour le profil principal
        $updateSql = "UPDATE profils_topographiques SET 
                        code_referent = ?,
                        numero_profil_borne = ?,
                        date_releve = ?,
                        heure_debut = ?,
                        latitude_borne = ?,
                        longitude_borne = ?,
                        altitude_borne = ?,
                        azimut = ?,
                        maree_type = ?,
                        evenements_extremes = ?,
                        autre_evenement = ?,
                        statut = ?
                    WHERE id = ?";
        
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->execute([
            $data['code_referent'] ?? '',
            $data['numero_profil_borne'] ?? '',
            $data['date_releve'] ?? '',
            $data['heure_debut'] ?? '',
            floatval($data['latitude_borne'] ?? 0),
            floatval($data['longitude_borne'] ?? 0),
            floatval($data['altitude_borne'] ?? 0),
            intval($data['azimut'] ?? 0),
            $data['maree_type'] ?? 'haute',
            is_array($data['evenements_extremes']) ? implode(',', $data['evenements_extremes']) : ($data['evenements_extremes'] ?? ''),
            $data['autre_evenement'] ?? '',
            $data['statut'] ?? 'brouillon',
            $profil_id
        ]);
        
        // Mettre à jour les mesures topographiques si fournies
        if (isset($data['mesures']) && is_array($data['mesures'])) {
            // Supprimer les anciennes mesures
            $deleteMesuresSql = "DELETE FROM mesures_topographiques WHERE profil_id = ?";
            $deleteMesuresStmt = $pdo->prepare($deleteMesuresSql);
            $deleteMesuresStmt->execute([$profil_id]);
            
            // Insérer les nouvelles mesures
            $insertMesureSql = "INSERT INTO mesures_topographiques 
                                (profil_id, numero_ordre, distance_metres, denivele_cm, denivele_cumule_cm, altitude_metres, commentaires) 
                                VALUES (?, ?, ?, ?, ?, ?, ?)";
            $insertMesureStmt = $pdo->prepare($insertMesureSql);
            
            $denivele_cumule = 0;
            foreach ($data['mesures'] as $mesure) {
                $denivele_cumule += floatval($mesure['denivele'] ?? 0);
                
                $insertMesureStmt->execute([
                    $profil_id,
                    intval($mesure['numero_ordre'] ?? 0),
                    floatval($mesure['distance'] ?? 0),
                    floatval($mesure['denivele'] ?? 0),
                    $denivele_cumule,
                    floatval($mesure['altitude'] ?? 0),
                    $mesure['commentaires'] ?? null
                ]);
            }
        }
        
        // Valider la transaction
        $pdo->commit();
        
        // Récupérer le profil mis à jour
        $updatedProfilSql = "SELECT 
                                p.*,
                                u.nom as referent_nom,
                                u.prenom as referent_prenom,
                                s.nom_site,
                                s.region
                            FROM profils_topographiques p
                            LEFT JOIN utilisateurs u ON p.utilisateur_id = u.id
                            LEFT JOIN sites s ON p.site_id = s.id
                            WHERE p.id = ?";
        
        $updatedProfilStmt = $pdo->prepare($updatedProfilSql);
        $updatedProfilStmt->execute([$profil_id]);
        $updatedProfil = $updatedProfilStmt->fetch(PDO::FETCH_ASSOC);
        
        jsonResponse(true, 'Profil mis à jour avec succès', [
            'profil' => $updatedProfil,
            'modifications' => [
                'updated_fields' => array_keys($data),
                'mesures_updated' => isset($data['mesures']) ? count($data['mesures']) : 0,
                'timestamp' => date('Y-m-d H:i:s')
            ]
        ]);
        
    } catch (Exception $e) {
        // Annuler la transaction en cas d'erreur
        $pdo->rollback();
        throw $e;
    }
    
} catch (PDOException $e) {
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), [
        'error_type' => 'PDO',
        'error_code' => $e->getCode()
    ], 500);
} catch (Exception $e) {
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), [
        'error_type' => 'General'
    ], 500);
}
?>