<?php
/**
 * API pour valider/rejeter un profil topographique
 * Emplacement : /api/profils/validate.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

try {
    // Vérifier que c'est une requête POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        jsonResponse(false, 'Méthode non autorisée. Utilisez POST.', null, 405);
    }
    
    // Récupérer les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        jsonResponse(false, 'Données JSON invalides', null, 400);
    }
    
    // Validation des paramètres requis
    $profil_id = isset($data['id']) ? intval($data['id']) : null;
    $statut = isset($data['statut']) ? trim($data['statut']) : null;
    $commentaire = isset($data['commentaire_validation']) ? trim($data['commentaire_validation']) : '';
    
    if (!$profil_id) {
        jsonResponse(false, 'ID du profil requis', null, 400);
    }
    
    if (!in_array($statut, ['valide', 'rejete'])) {
        jsonResponse(false, 'Statut invalide. Utilisez "valide" ou "rejete".', null, 400);
    }
    
    if ($statut === 'rejete' && empty($commentaire)) {
        jsonResponse(false, 'Un commentaire est obligatoire pour rejeter un profil', null, 400);
    }
    
    // Connexion à la base de données
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Vérifier que le profil existe
    $checkSql = "SELECT id, statut, code_referent FROM profils_topographiques WHERE id = ?";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$profil_id]);
    $profil = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$profil) {
        jsonResponse(false, 'Profil topographique non trouvé', null, 404);
    }
    
    // Vérifier que le profil peut être validé
    $statuts_modifiables = ['brouillon', 'soumis'];
    if (!in_array($profil['statut'], $statuts_modifiables)) {
        jsonResponse(false, 'Ce profil ne peut plus être modifié (statut: ' . $profil['statut'] . ')', null, 400);
    }
    
    // Commencer une transaction
    $pdo->beginTransaction();
    
    try {
        // Mettre à jour le statut du profil
        $updateSql = "UPDATE profils_topographiques 
                      SET statut = ?, 
                          commentaire_validation = ?, 
                          date_validation = NOW(),
                          validateur_id = ? 
                      WHERE id = ?";
        
        // Pour l'instant, validateur_id = 1 (vous pouvez récupérer l'ID du validateur via token)
        $validateur_id = 1; // À remplacer par l'ID réel du validateur
        
        $updateStmt = $pdo->prepare($updateSql);
        $success = $updateStmt->execute([$statut, $commentaire, $validateur_id, $profil_id]);
        
        if (!$success) {
            throw new Exception('Erreur lors de la mise à jour du profil');
        }
        
        // Optionnel : Enregistrer l'action dans un log de validation
        try {
            $logSql = "INSERT INTO validation_logs 
                       (type_element, element_id, ancien_statut, nouveau_statut, commentaire, validateur_id, created_at) 
                       VALUES ('profil', ?, ?, ?, ?, ?, NOW())";
            
            $logStmt = $pdo->prepare($logSql);
            $logStmt->execute([
                $profil_id, 
                $profil['statut'], 
                $statut, 
                $commentaire, 
                $validateur_id
            ]);
        } catch (Exception $logError) {
            // Log silencieux si la table validation_logs n'existe pas
            error_log("Impossible d'enregistrer le log de validation: " . $logError->getMessage());
        }
        
        // Valider la transaction
        $pdo->commit();
        
        // Réponse de succès
        $message = ($statut === 'valide') ? 
            'Profil validé avec succès' : 
            'Profil rejeté avec succès';
        
        jsonResponse(true, $message, [
            'profil_id' => $profil_id,
            'code_referent' => $profil['code_referent'],
            'ancien_statut' => $profil['statut'],
            'nouveau_statut' => $statut,
            'commentaire' => $commentaire,
            'date_validation' => date('Y-m-d H:i:s'),
            'validateur_id' => $validateur_id
        ]);
        
    } catch (Exception $e) {
        // Annuler la transaction en cas d'erreur
        $pdo->rollback();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Erreur PDO dans validate.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("Erreur générale dans validate.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}
?>