<?php
// api/dechets/bulk-validate.php - Validation en lot des relevés de déchets

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Méthode non autorisée']);
    exit();
}

try {
    require_once '../config/database.php';
    require_once '../config/auth_middleware.php';

    // Vérifier l'authentification
    $auth = new AuthMiddleware();
    $user = $auth->authenticate();

    if (!$user) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Non authentifié']);
        exit();
    }

    // Vérifier que seuls les contrôleurs peuvent valider
    if ($user['role'] !== 'controleur') {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'message' => 'Accès refusé - Seuls les Contrôleurs peuvent valider en lot',
            'user_role' => $user['role']
        ]);
        exit();
    }

    // Lire les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Données JSON invalides']);
        exit();
    }

    // Validation des paramètres requis
    if (!isset($data['ids']) || !is_array($data['ids']) || empty($data['ids'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Liste d\'IDs requise']);
        exit();
    }

    if (!isset($data['statut'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Statut requis']);
        exit();
    }

    $ids = array_map('intval', $data['ids']);
    $nouveau_statut = trim($data['statut']);
    $commentaire = isset($data['commentaire_validation']) ? trim($data['commentaire_validation']) : '';

    // Valider le statut
    $statuts_valides = ['valide', 'rejete'];
    if (!in_array($nouveau_statut, $statuts_valides)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Statut invalide. Utiliser: valide ou rejete']);
        exit();
    }

    // Commentaire obligatoire pour rejet
    if ($nouveau_statut === 'rejete' && empty($commentaire)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Commentaire obligatoire pour rejeter des relevés']);
        exit();
    }

    $db = new Database();
    $pdo = $db->getConnection();

    // Vérifier que tous les relevés existent et peuvent être validés par ce contrôleur
    $placeholders = str_repeat('?,', count($ids) - 1) . '?';
    $stmt = $pdo->prepare("
        SELECT dcr.id, dcr.statut, dcr.site_id, s.nom_site, u.nom as referent_nom
        FROM dechets_collecte_releves dcr
        LEFT JOIN sites s ON dcr.site_id = s.id  
        LEFT JOIN users u ON dcr.utilisateur_id = u.id
        WHERE dcr.id IN ($placeholders)
    ");
    $stmt->execute($ids);
    $dechets = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (count($dechets) !== count($ids)) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Certains relevés n\'existent pas']);
        exit();
    }

    // Vérifier les permissions de site pour chaque relevé
    $errors = [];
    $valid_ids = [];
    
    foreach ($dechets as $dechet) {
        // Vérifier que le contrôleur peut valider ce relevé (même site)
        if ($user['site_id'] && $dechet['site_id'] != $user['site_id']) {
            $errors[] = "Relevé #{$dechet['id']} : site non autorisé";
            continue;
        }
        
        // Vérifier que le relevé n'a pas déjà ce statut
        if ($dechet['statut'] === $nouveau_statut) {
            $errors[] = "Relevé #{$dechet['id']} : déjà {$nouveau_statut}";
            continue;
        }
        
        $valid_ids[] = $dechet['id'];
    }

    if (empty($valid_ids)) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'message' => 'Aucun relevé ne peut être validé',
            'errors' => $errors
        ]);
        exit();
    }

    // Commencer la transaction
    $pdo->beginTransaction();

    try {
        $processed_count = 0;
        $action = $nouveau_statut === 'valide' ? 'validation' : 'rejet';

        foreach ($valid_ids as $dechet_id) {
            // 1. Mettre à jour le statut du relevé
            $stmt = $pdo->prepare("
                UPDATE dechets_collecte_releves 
                SET statut = ?, 
                    date_validation = NOW(),
                    validateur_id = ?,
                    commentaire_validation = ?,
                    updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $nouveau_statut,
                $user['id'],
                $commentaire,
                $dechet_id
            ]);

            if ($stmt->rowCount() > 0) {
                $processed_count++;

                // 2. Enregistrer l'historique de validation
                $stmt = $pdo->prepare("
                    INSERT INTO validation_historique 
                    (type_element, element_id, action, utilisateur_id, commentaire, site_id, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, NOW())
                ");
                
                $dechet_info = array_first($dechets, function($d) use ($dechet_id) {
                    return $d['id'] == $dechet_id;
                });
                
                $stmt->execute([
                    'dechet',
                    $dechet_id,
                    $action,
                    $user['id'],
                    "Validation en lot: " . $commentaire,
                    $dechet_info['site_id']
                ]);
            }
        }

        // 3. Log de l'activité globale
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, details, ip_address, created_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $details = json_encode([
            'type' => 'bulk_validation_dechets',
            'total_requested' => count($ids),
            'total_processed' => $processed_count,
            'statut' => $nouveau_statut,
            'commentaire' => $commentaire,
            'ids_processed' => $valid_ids,
            'errors' => $errors
        ]);
        
        $stmt->execute([
            $user['id'],
            "Validation en lot de {$processed_count} relevé(s) de déchets",
            $details,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);

        // Valider la transaction
        $pdo->commit();

        $response = [
            'success' => true,
            'message' => "{$processed_count} relevé(s) de déchets " . 
                        ($nouveau_statut === 'valide' ? 'validé(s)' : 'rejeté(s)') . 
                        " avec succès",
            'data' => [
                'total_requested' => count($ids),
                'processed' => $processed_count,
                'statut' => $nouveau_statut,
                'validateur' => [
                    'id' => $user['id'],
                    'nom' => trim($user['prenom'] . ' ' . $user['nom']),
                    'role' => $user['role']
                ],
                'commentaire' => $commentaire,
                'errors' => $errors
            ]
        ];

        echo json_encode($response, JSON_UNESCAPED_UNICODE);

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }

} catch (PDOException $e) {
    error_log("Erreur validation en lot déchets: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur base de données lors de la validation en lot',
        'error_details' => $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log("Erreur validation en lot déchets: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur serveur lors de la validation en lot',
        'error_details' => $e->getMessage()
    ]);
}

// Fonction helper pour array_first (si pas disponible)
if (!function_exists('array_first')) {
    function array_first($array, $callback) {
        foreach ($array as $item) {
            if ($callback($item)) {
                return $item;
            }
        }
        return null;
    }
}
?>