<?php
/**
 * API pour récupérer tous les détails de déchets
 * Emplacement : /public_html/onl/api/dechets/details.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

/**
 * Fonction pour envoyer une réponse JSON standardisée
 */
function jsonResponse($success, $message, $data = null, $status = 200) {
    http_response_code($status);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

/**
 * Fonction pour obtenir le label d'une catégorie de déchet
 */
function getCategorieLabel($code) {
    $categories = [
        'PL' => 'Plastique',
        'CA' => 'Carton/Papier', 
        'TX' => 'Textile',
        'ME' => 'Métal',
        'VE' => 'Verre',
        'BO' => 'Bois',
        'AU' => 'Autres'
    ];
    return $categories[$code] ?? $code;
}

/**
 * Fonction pour obtenir le label d'une source de pollution
 */
function getSourcePollutionLabel($code) {
    $sources = [
        '1' => 'Déchets ménagers',
        '2' => 'Déchets industriels',
        '3' => 'Pêche',
        '4' => 'Sanitaire',
        '5' => 'Autres'
    ];
    return $sources[$code] ?? 'Non spécifié';
}

try {
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Paramètres optionnels
    $releve_id = isset($_GET['releve_id']) ? intval($_GET['releve_id']) : null;
    $categorie = isset($_GET['categorie']) ? trim($_GET['categorie']) : null;
    $source_pollution = isset($_GET['source_pollution']) ? trim($_GET['source_pollution']) : null;
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : null;
    $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
    
    // Construction de la requête SQL
    $sql = "SELECT 
                d.id,
                d.releve_id,
                d.categorie,
                d.sous_categorie,
                d.nombre_articles,
                d.poids,
                d.source_pollution,
                r.code_referent,
                r.date_releve,
                r.heure_debut,
                r.latitude_debut,
                r.longitude_debut,
                r.latitude_fin,
                r.longitude_fin,
                r.statut as releve_statut,
                s.nom_site,
                s.region,
                u.nom as referent_nom,
                u.prenom as referent_prenom
            FROM dechets_detail d
            LEFT JOIN releves_dechets r ON d.releve_id = r.id
            LEFT JOIN sites s ON r.site_id = s.id
            LEFT JOIN utilisateurs u ON r.utilisateur_id = u.id";
    
    $conditions = [];
    $params = [];
    
    // Filtrage par relevé si spécifié
    if ($releve_id) {
        $conditions[] = "d.releve_id = :releve_id";
        $params[':releve_id'] = $releve_id;
    }
    
    // Filtrage par catégorie si spécifié
    if ($categorie) {
        $conditions[] = "d.categorie = :categorie";
        $params[':categorie'] = $categorie;
    }
    
    // Filtrage par source de pollution si spécifié
    if ($source_pollution) {
        $conditions[] = "d.source_pollution = :source_pollution";
        $params[':source_pollution'] = $source_pollution;
    }
    
    // Ajouter les conditions WHERE si nécessaire
    if (!empty($conditions)) {
        $sql .= " WHERE " . implode(" AND ", $conditions);
    }
    
    // Ordonner par relevé puis par catégorie
    $sql .= " ORDER BY d.releve_id ASC, d.categorie ASC";
    
    // Ajouter la pagination si spécifiée
    if ($limit) {
        $sql .= " LIMIT :limit";
        if ($offset > 0) {
            $sql .= " OFFSET :offset";
        }
        $params[':limit'] = $limit;
        if ($offset > 0) {
            $params[':offset'] = $offset;
        }
    }
    
    // Exécuter la requête
    $stmt = $pdo->prepare($sql);
    
    // Bind des paramètres
    foreach ($params as $key => $value) {
        if (in_array($key, [':limit', ':offset', ':releve_id'])) {
            $stmt->bindValue($key, $value, PDO::PARAM_INT);
        } else {
            $stmt->bindValue($key, $value, PDO::PARAM_STR);
        }
    }
    
    $stmt->execute();
    $details = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Log pour debug
    error_log("Détails déchets récupérés: " . count($details));
    
    // Formater les résultats
    $results = [];
    foreach ($details as $detail) {
        $poids = floatval($detail['poids'] ?? 0);
        $nombre_articles = intval($detail['nombre_articles'] ?? 0);
        
        $results[] = [
            'id' => intval($detail['id']),
            'releve_id' => intval($detail['releve_id']),
            'categorie' => $detail['categorie'],
            'categorie_label' => getCategorieLabel($detail['categorie']),
            'sous_categorie' => $detail['sous_categorie'],
            'nombre_articles' => $nombre_articles,
            'poids' => $poids,
            'source_pollution' => $detail['source_pollution'],
            'source_pollution_label' => getSourcePollutionLabel($detail['source_pollution']),
            
            // Calculs automatiques
            'poids_moyen_par_article_g' => $nombre_articles > 0 ? round(($poids / $nombre_articles) * 1000, 2) : 0,
            'densite_kg_m2' => 0, // À calculer si surface disponible
            
            // Informations du relevé associé
            'releve_info' => [
                'code_referent' => $detail['code_referent'],
                'date_releve' => $detail['date_releve'],
                'heure_debut' => $detail['heure_debut'],
                'latitude_debut' => floatval($detail['latitude_debut'] ?? 0),
                'longitude_debut' => floatval($detail['longitude_debut'] ?? 0),
                'latitude_fin' => floatval($detail['latitude_fin'] ?? 0),
                'longitude_fin' => floatval($detail['longitude_fin'] ?? 0),
                'statut' => $detail['releve_statut'],
                'referent_nom' => trim(($detail['referent_prenom'] ?? '') . ' ' . ($detail['referent_nom'] ?? '')),
                'site_nom' => $detail['nom_site'],
                'region' => $detail['region']
            ]
        ];
    }
    
    // Calculer des statistiques
    $stats = [
        'total_details' => count($results),
        'releves_avec_details' => count(array_unique(array_column($results, 'releve_id'))),
        'categories_uniques' => count(array_unique(array_column($results, 'categorie'))),
        'poids_total_kg' => round(array_sum(array_column($results, 'poids')), 2),
        'articles_total' => array_sum(array_column($results, 'nombre_articles')),
        'derniere_mise_a_jour' => date('Y-m-d H:i:s')
    ];
    
    // Statistiques par catégorie
    $stats_categories = [];
    foreach ($results as $result) {
        $cat = $result['categorie'];
        if (!isset($stats_categories[$cat])) {
            $stats_categories[$cat] = [
                'categorie' => $cat,
                'label' => $result['categorie_label'],
                'occurrences' => 0,
                'poids_total' => 0,
                'articles_total' => 0
            ];
        }
        $stats_categories[$cat]['occurrences']++;
        $stats_categories[$cat]['poids_total'] += $result['poids'];
        $stats_categories[$cat]['articles_total'] += $result['nombre_articles'];
    }
    
    $stats['par_categorie'] = array_values($stats_categories);
    
    // Ajouter les paramètres de filtrage aux stats
    if ($releve_id) $stats['releve_filtre'] = $releve_id;
    if ($categorie) $stats['categorie_filtre'] = $categorie;
    if ($source_pollution) $stats['source_pollution_filtre'] = $source_pollution;
    
    // Réponse finale
    $response_data = [
        'details' => $results,
        'statistiques' => $stats,
        'parametres' => [
            'releve_id' => $releve_id,
            'categorie' => $categorie,
            'source_pollution' => $source_pollution,
            'limit' => $limit,
            'offset' => $offset
        ]
    ];
    
    jsonResponse(true, count($results) . ' détail(s) de déchets trouvé(s)', $response_data);
    
} catch (PDOException $e) {
    error_log("Erreur PDO dans details.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("Erreur générale dans details.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}
?>