<?php
/**
 * API pour récupérer les détails d'un relevé de déchets AVEC PHOTOS - VERSION CORRIGÉE SANS CONFLIT
 * Emplacement : /public_html/onl/api/dechets/get.php
 * Usage : GET /api/dechets/get.php?id=2
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

try {
    // Vérifier que l'ID est fourni
    if (!isset($_GET['id']) || empty($_GET['id'])) {
        jsonResponse(false, 'ID du relevé requis', null, 400);
        exit;
    }
    
    $releve_id = intval($_GET['id']);
    
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
        exit;
    }
    
    error_log("=== API dechets/get.php - ID: $releve_id ===");
    
    // Récupérer les informations principales du relevé
    $sql = "SELECT 
                r.*,
                u.nom as referent_nom,
                u.prenom as referent_prenom,
                u.email as referent_email,
                u.telephone as referent_telephone,
                s.nom_site,
                s.region,
                s.latitude as site_latitude,
                s.longitude as site_longitude
            FROM releves_dechets r
            LEFT JOIN utilisateurs u ON r.utilisateur_id = u.id
            LEFT JOIN sites s ON r.site_id = s.id
            WHERE r.id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$releve_id]);
    $releve = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$releve) {
        jsonResponse(false, 'Relevé de déchets non trouvé', null, 404);
        exit;
    }
    
    // === REQUÊTE AMÉLIORÉE POUR DEBUG DES PHOTOS ===
    $sql_details = "SELECT 
                        id,
                        releve_id,
                        categorie,
                        sous_categorie,
                        nombre_articles,
                        poids,
                        source_pollution,
                        commentaires,
                        has_photo,
                        photo_filename,
                        photo_path,
                        photo_size,
                        photo_uploaded_at,
                        photo,
                        LENGTH(photo) as photo_blob_size,
                        CASE 
                            WHEN photo IS NULL THEN 'NULL'
                            WHEN photo = '' THEN 'EMPTY'
                            WHEN LENGTH(photo) = 0 THEN 'ZERO_LENGTH'
                            WHEN LENGTH(photo) > 0 THEN 'HAS_DATA'
                            ELSE 'UNKNOWN'
                        END as photo_status
                    FROM dechets_detail 
                    WHERE releve_id = ? 
                    ORDER BY id";
    
    $stmt_details = $pdo->prepare($sql_details);
    $stmt_details->execute([$releve_id]);
    $details_dechets = $stmt_details->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("Nombre de déchets récupérés: " . count($details_dechets));
    
    // Debug de chaque ligne brute
    foreach ($details_dechets as $index => $detail) {
        error_log("=== DÉCHET BRUT $index (ID: {$detail['id']}) ===");
        error_log("Catégorie: " . $detail['categorie']);
        error_log("has_photo: " . var_export($detail['has_photo'], true));
        error_log("photo_status: " . $detail['photo_status']);
        error_log("photo_blob_size: " . $detail['photo_blob_size']);
        error_log("photo_filename: " . ($detail['photo_filename'] ?? 'NULL'));
        error_log("photo_size: " . ($detail['photo_size'] ?? 'NULL'));
    }
    
    // Calculer les statistiques
    $total_quantite = 0;
    $total_articles = 0;
    $categories = [];
    $photos_valides_count = 0;
    
    // Mapping des catégories
    $categorie_names = [
        'PL' => 'Plastique',
        'CA' => 'Carton/Papier', 
        'TX' => 'Textile',
        'ME' => 'Métal',
        'VE' => 'Verre',
        'BO' => 'Bois',
        'AU' => 'Autre'
    ];
    
    // Mapping des sous-catégories
    $sous_categorie_names = [
        'PL-01' => 'Bouteilles plastique',
        'PL-02' => 'Sacs plastique',
        'PL-03' => 'Emballages plastique',
        'CA-01' => 'Cartons alimentaires',
        'CA-02' => 'Papiers divers',
        'TX-01' => 'Vêtements',
        'ME-01' => 'Canettes métalliques',
        'VE-01' => 'Bouteilles verre',
        'BO-01' => 'Bois traité',
        'AU-01' => 'Autres déchets'
    ];
    
    // Mapping des sources de pollution - CORRIGÉ
    $sources_pollution = [
        '1' => 'Alimentaire',
        '2' => 'Tabac',
        '3' => 'Pêche',
        '4' => 'Conchyliculture',
        '5' => 'Sanitaire',
        '6' => 'Autres'
    ];
    
    // Mapping des codes d'orientation, courant, vent
    $orientation_map = [
        '01' => 'Nord', '02' => 'Nord-Est', '03' => 'Est', '04' => 'Sud-Est',
        '05' => 'Sud', '06' => 'Sud-Ouest', '07' => 'Ouest', '08' => 'Nord-Ouest'
    ];
    
    // === FORMATER LES DÉTAILS DES DÉCHETS AVEC DEBUG MAXIMAL ===
    $dechets_formattes = [];
    
    foreach ($details_dechets as $detail) {
        // Calculer les statistiques de base
        $total_quantite += floatval($detail['poids']);
        $total_articles += intval($detail['nombre_articles']);
        if (!in_array($detail['categorie'], $categories)) {
            $categories[] = $detail['categorie'];
        }
        
        $categorie_nom = isset($categorie_names[$detail['categorie']]) ? 
                        $categorie_names[$detail['categorie']] : 
                        $detail['categorie'];
        
        $sous_categorie_nom = isset($sous_categorie_names[$detail['sous_categorie']]) ? 
                             $sous_categorie_names[$detail['sous_categorie']] : 
                             $detail['sous_categorie'];
        
        $source_nom = isset($sources_pollution[$detail['source_pollution']]) ? 
                     $sources_pollution[$detail['source_pollution']] : 
                     'Autre';
        
        // === TRAITEMENT MAXIMAL DE DEBUG DES PHOTOS ===
        $photo_base64 = null;
        $has_valid_photo = false;
        
        error_log("=== TRAITEMENT PHOTO DÉCHET ID {$detail['id']} ===");
        error_log("Catégorie: {$detail['categorie']} ({$categorie_nom})");
        error_log("has_photo (valeur brute): " . var_export($detail['has_photo'], true));
        error_log("has_photo (type): " . gettype($detail['has_photo']));
        error_log("photo_status: " . $detail['photo_status']);
        error_log("photo_blob_size (SQL LENGTH): " . $detail['photo_blob_size']);
        
        // Tests multiples de détection de photo
        $photo_tests = [
            'is_not_null' => !is_null($detail['photo']),
            'is_not_empty_string' => $detail['photo'] !== '',
            'strlen_gt_0' => strlen($detail['photo'] ?? '') > 0,
            'strlen_gt_100' => strlen($detail['photo'] ?? '') > 100,
            'sql_length_gt_0' => $detail['photo_blob_size'] > 0,
            'photo_status_has_data' => $detail['photo_status'] === 'HAS_DATA'
        ];
        
        error_log("Tests photo: " . json_encode($photo_tests));
        
        // Condition la plus permissive possible pour commencer
        if ($detail['photo_status'] === 'HAS_DATA' && $detail['photo_blob_size'] > 0) {
            try {
                error_log("🔄 Tentative d'encodage base64...");
                
                // Convertir le BLOB en base64
                $photo_base64 = base64_encode($detail['photo']);
                
                if ($photo_base64 && strlen($photo_base64) > 50) {
                    $has_valid_photo = true;
                    $photos_valides_count++;
                    
                    error_log("✅ Photo valide encodée !");
                    error_log("   Taille base64: " . strlen($photo_base64) . " caractères");
                    error_log("   Début base64: " . substr($photo_base64, 0, 50) . "...");
                    error_log("   Type détecté: " . (substr($photo_base64, 0, 10) === '/9j/' ? 'JPEG' : 'Autre'));
                } else {
                    error_log("❌ Base64 généré mais invalide ou trop court");
                    error_log("   Taille: " . strlen($photo_base64) . " caractères");
                    $photo_base64 = null;
                    $has_valid_photo = false;
                }
            } catch (Exception $e) {
                error_log("❌ Erreur lors de l'encodage: " . $e->getMessage());
                $photo_base64 = null;
                $has_valid_photo = false;
            }
        } else {
            error_log("❌ Conditions de base non remplies pour la photo");
            error_log("   photo_status: " . $detail['photo_status']);
            error_log("   photo_blob_size: " . $detail['photo_blob_size']);
        }
        
        // Si toujours pas de photo valide, essayer une approche alternative
        if (!$has_valid_photo && !empty($detail['photo'])) {
            error_log("🔄 Tentative alternative d'encodage...");
            try {
                $photo_base64_alt = base64_encode($detail['photo']);
                if ($photo_base64_alt && strlen($photo_base64_alt) > 10) {
                    error_log("⚠️  Photo alternative trouvée (taille: " . strlen($photo_base64_alt) . ")");
                    // Utiliser même si c'est court, pour debug
                    $photo_base64 = $photo_base64_alt;
                    $has_valid_photo = true;
                    $photos_valides_count++;
                }
            } catch (Exception $e) {
                error_log("❌ Erreur encodage alternatif: " . $e->getMessage());
            }
        }
        
        error_log("Résultat final pour déchet {$detail['id']}: " . ($has_valid_photo ? 'PHOTO OK' : 'PAS DE PHOTO'));
        error_log("---");
        
        // Ajouter le déchet formaté à la liste
        $dechets_formattes[] = [
            'id' => intval($detail['id']),
            'categorie' => $detail['categorie'],
            'categorie_nom' => $categorie_nom,
            'sous_categorie' => $detail['sous_categorie'] ?? '',
            'sous_categorie_nom' => $sous_categorie_nom ?? '',
            'nombre_articles' => intval($detail['nombre_articles']),
            'poids' => floatval($detail['poids']),
            'source_pollution' => $detail['source_pollution'],
            'source_pollution_nom' => $source_nom,
            'commentaires' => $detail['commentaires'] ?? '',
            
            // === PHOTO EN BASE64 ===
            'photo' => $photo_base64,
            'has_photo' => $has_valid_photo,
            'photo_filename' => $detail['photo_filename'] ?? null,
            'photo_size' => intval($detail['photo_size'] ?? 0),
            'photo_uploaded_at' => $detail['photo_uploaded_at'] ?? null,
            'photo_blob_size' => intval($detail['photo_blob_size'] ?? 0),
            'photo_status' => $detail['photo_status'],
            
            // DEBUG : Ajouter les tests pour vérification côté client
            'debug_tests' => $photo_tests ?? []
        ];
    }
    
    // Log des statistiques finales
    error_log("=== STATISTIQUES FINALES ===");
    error_log("Total déchets: " . count($dechets_formattes));
    error_log("Photos valides trouvées: $photos_valides_count");
    error_log("Pourcentage avec photos: " . (count($dechets_formattes) > 0 ? round(($photos_valides_count / count($dechets_formattes)) * 100, 1) : 0) . "%");
    
    // Construire la réponse complète
    $response_data = [
        // Section I: Informations d'identification
        'identification' => [
            'id' => intval($releve['id']),
            'code_referent' => $releve['code_referent'],
            'date_releve' => $releve['date_releve'],
            'heure_debut' => $releve['heure_debut'],
            'statut' => $releve['statut'],
            'created_at' => $releve['created_at']
        ],
        
        // Informations du référent
        'referent' => [
            'nom_complet' => trim(($releve['referent_prenom'] ?: '') . ' ' . ($releve['referent_nom'] ?: '')),
            'nom' => $releve['referent_nom'],
            'prenom' => $releve['referent_prenom'],
            'email' => $releve['referent_email'],
            'telephone' => $releve['referent_telephone']
        ],
        
        // Informations du site
        'site' => [
            'nom_site' => $releve['nom_site'],
            'region' => $releve['region'],
            'latitude' => floatval($releve['site_latitude'] ?: 0),
            'longitude' => floatval($releve['site_longitude'] ?: 0)
        ],
        
        // Section II: Position GPS et conditions
        'position_conditions' => [
            'latitude_debut' => floatval($releve['latitude_debut'] ?: 0),
            'longitude_debut' => floatval($releve['longitude_debut'] ?: 0),
            'latitude_fin' => floatval($releve['latitude_fin'] ?: 0),
            'longitude_fin' => floatval($releve['longitude_fin'] ?: 0),
            'orientation_plage' => $releve['orientation_plage'],
            'orientation_nom' => isset($orientation_map[$releve['orientation_plage']]) ? 
                               $orientation_map[$releve['orientation_plage']] : 
                               $releve['orientation_plage'],
            'courant_dominant' => $releve['courant_dominant'],
            'courant_nom' => isset($orientation_map[$releve['courant_dominant']]) ? 
                           $orientation_map[$releve['courant_dominant']] : 
                           $releve['courant_dominant'],
            'vent_dominant' => $releve['vent_dominant'],
            'vent_nom' => isset($orientation_map[$releve['vent_dominant']]) ? 
                        $orientation_map[$releve['vent_dominant']] : 
                        $releve['vent_dominant']
        ],
        
        // Section III: Événements extrêmes
        'evenements_extremes' => [
            'liste_brute' => $releve['evenements_extremes'],
            'autre_evenement' => $releve['autre_evenement'],
            'nombre_evenements' => !empty($releve['evenements_extremes']) ? 1 : 0
        ],
        
        // Section IV: Détails des déchets collectés AVEC PHOTOS
        'dechets_detail' => [
            'liste' => $dechets_formattes,
            'nombre_types' => count($dechets_formattes),
            'categories_presentes' => $categories,
            'photos_statistiques' => [
                'total_photos' => $photos_valides_count,
                'dechets_avec_photos' => $photos_valides_count,
                'dechets_sans_photos' => count($dechets_formattes) - $photos_valides_count,
                'pourcentage_avec_photos' => count($dechets_formattes) > 0 ? round(($photos_valides_count / count($dechets_formattes)) * 100, 1) : 0
            ]
        ],
        
        // Statistiques générales
        'statistiques' => [
            'total_quantite' => $total_quantite,
            'total_articles' => $total_articles,
            'nombre_categories' => count($categories),
            'nombre_details' => count($dechets_formattes),
            'total_photos' => $photos_valides_count
        ]
    ];
    
    error_log("✅ Réponse formatée avec " . count($dechets_formattes) . " déchets, dont " . $photos_valides_count . " avec photos valides");
    
    jsonResponse(true, 'Détails du relevé récupérés avec succès', $response_data);
    
} catch (PDOException $e) {
    error_log("❌ Erreur PDO dans dechets/get.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("❌ Erreur générale dans dechets/get.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}

/**
 * Fonction utilitaire pour formater la taille des fichiers
 */
function formatFileSize($bytes) {
    if ($bytes == 0) return '0 B';
    
    $k = 1024;
    $sizes = ['B', 'KB', 'MB', 'GB'];
    $i = floor(log($bytes) / log($k));
    
    return round($bytes / pow($k, $i), 2) . ' ' . $sizes[$i];
}

// REMARQUE: La fonction jsonResponse() est déjà définie dans config/database.php
// donc nous n'ajoutons pas de redéclaration ici
?>