<?php
/**
 * API pour lister les relevés de déchets - Données brutes du terrain
 * Emplacement : /public_html/onl/api/dechets/list.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

try {
    // Utiliser la classe Database existante
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
        exit;
    }
    
    error_log("=== API dechets/list.php - Données brutes du terrain ===");
    
    // Requête simple - EXACTEMENT les données de releves_dechets
    $sql = "SELECT 
                r.id,
                r.code_referent,
                r.date_releve,
                r.heure_debut,
                r.latitude_debut,
                r.longitude_debut,
                r.latitude_fin,
                r.longitude_fin,
                r.orientation_plage,
                r.courant_dominant,
                r.vent_dominant,
                r.evenements_extremes,
                r.autre_evenement,
                r.statut,
                r.created_at,
                r.utilisateur_id,
                r.site_id,
                u.nom as referent_nom,
                u.prenom as referent_prenom,
                s.nom_site,
                s.region
            FROM releves_dechets r
            LEFT JOIN utilisateurs u ON r.utilisateur_id = u.id
            LEFT JOIN sites s ON r.site_id = s.id
            ORDER BY r.created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $releves = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("Nombre de relevés bruts: " . count($releves));
    
    // Formatage minimal - SANS calculs
    $results = [];
    foreach ($releves as $releve) {
        $results[] = [
            'id' => intval($releve['id']),
            'code_referent' => $releve['code_referent'] ?: '',
            'date_releve' => $releve['date_releve'] ?: '',
            'date_collecte' => $releve['date_releve'] ?: '', // Alias pour le frontend
            'heure_debut' => $releve['heure_debut'] ?: '',
            'latitude_debut' => floatval($releve['latitude_debut'] ?: 0),
            'longitude_debut' => floatval($releve['longitude_debut'] ?: 0),
            'latitude_fin' => floatval($releve['latitude_fin'] ?: 0),
            'longitude_fin' => floatval($releve['longitude_fin'] ?: 0),
            'orientation_plage' => $releve['orientation_plage'] ?: '',
            'courant_dominant' => $releve['courant_dominant'] ?: '',
            'vent_dominant' => $releve['vent_dominant'] ?: '',
            'evenements_extremes' => $releve['evenements_extremes'] ?: '',
            'autre_evenement' => $releve['autre_evenement'] ?: '',
            'statut' => $releve['statut'] ?: 'brouillon',
            'referent_nom' => trim(($releve['referent_prenom'] ?: '') . ' ' . ($releve['referent_nom'] ?: '')),
            'nom_site' => $releve['nom_site'] ?: '',
            'site_nom' => $releve['nom_site'] ?: '', // Alias pour le frontend
            'region' => $releve['region'] ?: '',
            'created_at' => $releve['created_at'] ?: '',
            'utilisateur_id' => intval($releve['utilisateur_id'] ?: 0),
            'site_id' => intval($releve['site_id'] ?: 0),
            // Champs vides pour compatibilité frontend mais sans calculs
            'type_principal' => 'Données terrain',
            'quantite_totale' => 'N/A'
        ];
    }
    
    error_log("Résultats formatés: " . count($results));
    
    // Réponse finale
    $response_data = [
        'releves' => $results,
        'total' => count($results)
    ];
    
    jsonResponse(true, count($results) . ' relevé(s) de déchets trouvé(s)', $response_data);
    
} catch (PDOException $e) {
    error_log("Erreur PDO: " . $e->getMessage());
    jsonResponse(false, 'Erreur de base de données: ' . $e->getMessage(), null, 500);
} catch (Exception $e) {
    error_log("Erreur générale: " . $e->getMessage());
    jsonResponse(false, 'Erreur serveur: ' . $e->getMessage(), null, 500);
}
?>