<?php
/**
 * Fichier de test pour débugger l'API dechets/list.php
 * Emplacement : /public_html/onl/api/dechets/test_list.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: text/html; charset=UTF-8');

echo "<!DOCTYPE html>";
echo "<html><head><meta charset='UTF-8'><title>Test Debug - API Déchets</title>";
echo "<style>
    body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
    .container { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
    table { width: 100%; border-collapse: collapse; margin: 10px 0; }
    th, td { padding: 8px; text-align: left; border: 1px solid #ddd; }
    th { background-color: #f2f2f2; font-weight: bold; }
    .success { color: #28a745; }
    .error { color: #dc3545; }
    .warning { color: #ffc107; }
    .info { color: #17a2b8; }
    .section { margin: 20px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
    pre { background: #f8f9fa; padding: 10px; border-radius: 4px; overflow-x: auto; }
</style>";
echo "</head><body>";

echo "<div class='container'>";
echo "<h1>🔍 Test Debug - API Déchets List</h1>";
echo "<p><em>Diagnostic complet de l'API dechets/list.php</em></p>";
echo "<hr>";

// Inclusion du fichier de configuration database existant
require_once '../config/database.php';

try {
    echo "<div class='section'>";
    echo "<h2>1. Test de connexion à la base de données</h2>";
    
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        echo "<p class='error'>❌ <strong>Erreur:</strong> Impossible de se connecter à la base de données</p>";
        exit;
    }
    
    echo "<p class='success'>✅ <strong>Connexion réussie</strong> à la base de données</p>";
    
    // Test info base
    $info = $database->testConnection();
    if ($info) {
        echo "<p class='info'>📋 <strong>Infos DB:</strong> Version: {$info['version']}, Base: {$info['database']}, User: {$info['user']}</p>";
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>2. Vérification des tables</h2>";
    
    // Vérifier si la table releves_dechets existe
    $tables_check = $pdo->query("SHOW TABLES LIKE 'releves_dechets'")->fetchAll();
    if (empty($tables_check)) {
        echo "<p class='error'>❌ <strong>Erreur:</strong> Table 'releves_dechets' non trouvée</p>";
        exit;
    }
    echo "<p class='success'>✅ <strong>Table trouvée:</strong> releves_dechets</p>";
    
    // Vérifier la table dechets_detail
    $details_table_check = $pdo->query("SHOW TABLES LIKE 'dechets_detail'")->fetchAll();
    if (empty($details_table_check)) {
        echo "<p class='warning'>⚠️ <strong>Attention:</strong> Table 'dechets_detail' non trouvée</p>";
    } else {
        echo "<p class='success'>✅ <strong>Table trouvée:</strong> dechets_detail</p>";
    }
    
    // Vérifier les autres tables
    $tables = ['utilisateurs', 'sites'];
    foreach ($tables as $table) {
        $check = $pdo->query("SHOW TABLES LIKE '$table'")->fetchAll();
        if (empty($check)) {
            echo "<p class='warning'>⚠️ <strong>Attention:</strong> Table '$table' non trouvée</p>";
        } else {
            echo "<p class='success'>✅ <strong>Table trouvée:</strong> $table</p>";
        }
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>3. Contenu de la table releves_dechets</h2>";
    
    // Compter les enregistrements
    $count = $pdo->query("SELECT COUNT(*) as total FROM releves_dechets")->fetch();
    echo "<p class='info'>📊 <strong>Nombre d'enregistrements:</strong> {$count['total']}</p>";
    
    if ($count['total'] > 0) {
        $releves = $pdo->query("SELECT * FROM releves_dechets ORDER BY created_at DESC LIMIT 5")->fetchAll();
        
        echo "<table>";
        echo "<tr><th>ID</th><th>Code Référent</th><th>Date Relevé</th><th>Site ID</th><th>Utilisateur ID</th><th>Statut</th><th>Créé le</th></tr>";
        
        foreach ($releves as $releve) {
            echo "<tr>";
            echo "<td>{$releve['id']}</td>";
            echo "<td>{$releve['code_referent']}</td>";
            echo "<td>{$releve['date_releve']}</td>";
            echo "<td>{$releve['site_id']}</td>";
            echo "<td>{$releve['utilisateur_id']}</td>";
            echo "<td>{$releve['statut']}</td>";
            echo "<td>{$releve['created_at']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        if ($count['total'] > 5) {
            echo "<p><em>... et " . ($count['total'] - 5) . " autre(s) enregistrement(s)</em></p>";
        }
    } else {
        echo "<p class='warning'>ℹ️ <strong>Aucun relevé trouvé dans la table</strong></p>";
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>4. Test des JOIN avec utilisateurs et sites</h2>";
    
    $sql_join = "SELECT 
                    r.id,
                    r.code_referent,
                    r.date_releve,
                    r.statut,
                    u.nom as referent_nom,
                    u.prenom as referent_prenom,
                    s.nom_site,
                    s.region
                FROM releves_dechets r
                LEFT JOIN utilisateurs u ON r.utilisateur_id = u.id
                LEFT JOIN sites s ON r.site_id = s.id
                ORDER BY r.created_at DESC
                LIMIT 3";
    
    try {
        $stmt_join = $pdo->prepare($sql_join);
        $stmt_join->execute();
        $releves_join = $stmt_join->fetchAll();
        
        echo "<p class='info'>📊 <strong>Résultats avec JOIN:</strong> " . count($releves_join) . " enregistrement(s)</p>";
        
        if (!empty($releves_join)) {
            echo "<table>";
            echo "<tr><th>ID</th><th>Code Référent</th><th>Date</th><th>Référent</th><th>Site</th><th>Région</th><th>Statut</th></tr>";
            
            foreach ($releves_join as $releve) {
                echo "<tr>";
                echo "<td>{$releve['id']}</td>";
                echo "<td>{$releve['code_referent']}</td>";
                echo "<td>{$releve['date_releve']}</td>";
                echo "<td>" . trim(($releve['referent_prenom'] ?? '') . ' ' . ($releve['referent_nom'] ?? '')) . "</td>";
                echo "<td>{$releve['nom_site']}</td>";
                echo "<td>{$releve['region']}</td>";
                echo "<td>{$releve['statut']}</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p class='warning'>ℹ️ <strong>Aucun relevé avec JOIN trouvé</strong></p>";
        }
    } catch (Exception $e) {
        echo "<p class='error'>❌ <strong>Erreur JOIN:</strong> " . $e->getMessage() . "</p>";
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>5. Test de la table dechets_detail</h2>";
    
    if (!empty($details_table_check)) {
        $details_count = $pdo->query("SELECT COUNT(*) as total FROM dechets_detail")->fetch();
        echo "<p class='info'>📊 <strong>Nombre de détails de déchets:</strong> {$details_count['total']}</p>";
        
        if ($details_count['total'] > 0) {
            $details = $pdo->query("SELECT d.*, r.code_referent FROM dechets_detail d LEFT JOIN releves_dechets r ON d.releve_id = r.id LIMIT 10")->fetchAll();
            
            echo "<table>";
            echo "<tr><th>ID</th><th>Relevé ID</th><th>Code Référent</th><th>Catégorie</th><th>Sous-catégorie</th><th>Articles</th><th>Poids (kg)</th></tr>";
            
            foreach ($details as $detail) {
                echo "<tr>";
                echo "<td>{$detail['id']}</td>";
                echo "<td>{$detail['releve_id']}</td>";
                echo "<td>{$detail['code_referent']}</td>";
                echo "<td>{$detail['categorie']}</td>";
                echo "<td>{$detail['sous_categorie']}</td>";
                echo "<td>{$detail['nombre_articles']}</td>";
                echo "<td>{$detail['poids']}</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<p class='warning'>ℹ️ <strong>Aucun détail de déchets trouvé</strong></p>";
        }
    } else {
        echo "<p class='error'>❌ <strong>Table dechets_detail non disponible</strong></p>";
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>6. Test de la requête complète (comme dans l'API)</h2>";
    
    $api_sql = "SELECT 
                    r.id,
                    r.code_referent,
                    r.date_releve,
                    r.heure_debut,
                    r.latitude_debut,
                    r.longitude_debut,
                    r.latitude_fin,
                    r.longitude_fin,
                    r.orientation_plage,
                    r.courant_dominant,
                    r.vent_dominant,
                    r.evenements_extremes,
                    r.autre_evenement,
                    r.statut,
                    r.created_at,
                    r.utilisateur_id,
                    r.site_id,
                    u.nom as referent_nom,
                    u.prenom as referent_prenom,
                    s.nom_site,
                    s.region
                FROM releves_dechets r
                LEFT JOIN utilisateurs u ON r.utilisateur_id = u.id
                LEFT JOIN sites s ON r.site_id = s.id
                ORDER BY r.created_at DESC";
    
    try {
        $api_stmt = $pdo->prepare($api_sql);
        $api_stmt->execute();
        $api_releves = $api_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo "<p class='info'>📊 <strong>Résultats de la requête API:</strong> " . count($api_releves) . " enregistrement(s)</p>";
        
        if (!empty($api_releves)) {
            echo "<p class='success'>✅ <strong>Requête API fonctionnelle</strong></p>";
            echo "<p><strong>Premier résultat:</strong></p>";
            echo "<pre>" . htmlspecialchars(json_encode($api_releves[0], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) . "</pre>";
            
            // Test des détails pour le premier relevé
            if (!empty($details_table_check)) {
                $first_id = $api_releves[0]['id'];
                $details_stmt = $pdo->prepare("SELECT COUNT(*) as nb_dechets, 
                                                     COALESCE(SUM(poids), 0) as total_quantite, 
                                                     COALESCE(SUM(nombre_articles), 0) as total_articles,
                                                     GROUP_CONCAT(DISTINCT categorie ORDER BY categorie) as categories
                                              FROM dechets_detail WHERE releve_id = ?");
                $details_stmt->execute([$first_id]);
                $details_result = $details_stmt->fetch(PDO::FETCH_ASSOC);
                
                echo "<p><strong>Détails des déchets pour le relevé #{$first_id}:</strong></p>";
                echo "<pre>" . htmlspecialchars(json_encode($details_result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) . "</pre>";
            }
        } else {
            echo "<p class='error'>❌ <strong>Aucun résultat avec la requête API</strong></p>";
        }
    } catch (Exception $e) {
        echo "<p class='error'>❌ <strong>Erreur requête API:</strong> " . $e->getMessage() . "</p>";
    }
    echo "</div>";
    
    echo "<div class='section'>";
    echo "<h2>7. Test de l'API complète (simulation JSON)</h2>";
    
    try {
        // Capturer la sortie de l'API
        ob_start();
        
        // Simuler l'inclusion de l'API (sans les headers pour éviter les conflits)
        $api_test_content = file_get_contents(__DIR__ . '/list.php');
        // Enlever les headers pour éviter les conflits
        $api_test_content = preg_replace('/header\([^)]+\);/', '', $api_test_content);
        $api_test_content = preg_replace('/if \(\$_SERVER\[\'REQUEST_METHOD\'\] === \'OPTIONS\'\).*?exit;/s', '', $api_test_content);
        
        eval('?>' . $api_test_content);
        
        $api_output = ob_get_clean();
        
        echo "<p class='success'>✅ <strong>Test API réussi</strong></p>";
        echo "<p><strong>Sortie JSON de l'API:</strong></p>";
        echo "<pre style='max-height: 400px; overflow-y: auto;'>" . htmlspecialchars($api_output) . "</pre>";
        
    } catch (Exception $e) {
        ob_end_clean();
        echo "<p class='error'>❌ <strong>Erreur lors du test API:</strong> " . $e->getMessage() . "</p>";
    }
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='section'>";
    echo "<h2 class='error'>❌ Erreur générale</h2>";
    echo "<p><strong>Message:</strong> " . $e->getMessage() . "</p>";
    echo "<p><strong>Fichier:</strong> " . $e->getFile() . "</p>";
    echo "<p><strong>Ligne:</strong> " . $e->getLine() . "</p>";
    echo "<p><strong>Trace:</strong></p>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
    echo "</div>";
}

echo "<div class='section'>";
echo "<h2>🔗 Liens utiles</h2>";
echo "<ul>";
echo "<li><a href='list.php' target='_blank'>Tester l'API list.php directement</a></li>";
echo "<li><a href='../profils/list.php' target='_blank'>Tester l'API profils/list.php (pour comparaison)</a></li>";
echo "<li><a href='../../web/index.html' target='_blank'>Retour à l'interface principale</a></li>";
echo "<li><a href='https://dgidbail.sn/onl/web/index.html' target='_blank'>Interface principale (lien direct)</a></li>";
echo "</ul>";
echo "</div>";

echo "<div class='section'>";
echo "<h2>📋 Résumé du diagnostic</h2>";
echo "<p><strong>Statut:</strong> Test terminé le " . date('d/m/Y à H:i:s') . "</p>";
echo "<p><em>Ce fichier de test vous aide à identifier les problèmes dans votre API dechets/list.php</em></p>";
echo "</div>";

echo "</div>"; // fin container
echo "</body></html>";
?>