<?php
// api/dechets/validate.php - Validation des relevés de déchets

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Méthode non autorisée']);
    exit();
}

try {
    require_once '../config/database.php';
    require_once '../config/auth_middleware.php';

    // Vérifier l'authentification
    $auth = new AuthMiddleware();
    $user = $auth->authenticate();

    if (!$user) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Non authentifié']);
        exit();
    }

    // Vérifier que seuls les contrôleurs peuvent valider
    if ($user['role'] !== 'controleur') {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'message' => 'Accès refusé - Seuls les Contrôleurs peuvent valider les relevés de déchets',
            'user_role' => $user['role']
        ]);
        exit();
    }

    // Lire les données JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Données JSON invalides']);
        exit();
    }

    // Validation des paramètres requis
    if (!isset($data['id']) || !isset($data['statut'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'ID et statut requis']);
        exit();
    }

    $dechet_id = (int)$data['id'];
    $nouveau_statut = trim($data['statut']);
    $commentaire = isset($data['commentaire_validation']) ? trim($data['commentaire_validation']) : '';

    // Valider le statut
    $statuts_valides = ['valide', 'rejete'];
    if (!in_array($nouveau_statut, $statuts_valides)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Statut invalide. Utiliser: valide ou rejete']);
        exit();
    }

    // Commentaire obligatoire pour rejet
    if ($nouveau_statut === 'rejete' && empty($commentaire)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Commentaire obligatoire pour rejeter un relevé']);
        exit();
    }

    $db = new Database();
    $pdo = $db->getConnection();

    // Vérifier que le relevé existe et récupérer ses infos
    $stmt = $pdo->prepare("
        SELECT dcr.*, s.nom_site, s.region, u.nom as referent_nom, u.prenom as referent_prenom
        FROM dechets_collecte_releves dcr
        LEFT JOIN sites s ON dcr.site_id = s.id  
        LEFT JOIN users u ON dcr.utilisateur_id = u.id
        WHERE dcr.id = ?
    ");
    $stmt->execute([$dechet_id]);
    $dechet = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$dechet) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Relevé de déchets non trouvé']);
        exit();
    }

    // Vérifier que le contrôleur peut valider ce relevé (même site)
    if ($user['site_id'] && $dechet['site_id'] != $user['site_id']) {
        http_response_code(403);
        echo json_encode([
            'success' => false, 
            'message' => 'Vous ne pouvez valider que les relevés de votre site',
            'user_site_id' => $user['site_id'],
            'dechet_site_id' => $dechet['site_id']
        ]);
        exit();
    }

    // Vérifier le statut actuel
    if ($dechet['statut'] === $nouveau_statut) {
        echo json_encode([
            'success' => true, 
            'message' => 'Le relevé a déjà ce statut',
            'current_status' => $dechet['statut']
        ]);
        exit();
    }

    // Commencer la transaction
    $pdo->beginTransaction();

    try {
        // 1. Mettre à jour le statut du relevé
        $stmt = $pdo->prepare("
            UPDATE dechets_collecte_releves 
            SET statut = ?, 
                date_validation = NOW(),
                validateur_id = ?,
                commentaire_validation = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $stmt->execute([
            $nouveau_statut,
            $user['id'],
            $commentaire,
            $dechet_id
        ]);

        // 2. Enregistrer l'historique de validation
        $stmt = $pdo->prepare("
            INSERT INTO validation_historique 
            (type_element, element_id, action, utilisateur_id, commentaire, site_id, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $action = $nouveau_statut === 'valide' ? 'validation' : 'rejet';
        $stmt->execute([
            'dechet',
            $dechet_id,
            $action,
            $user['id'],
            $commentaire,
            $dechet['site_id']
        ]);

        // 3. Log de l'activité
        $stmt = $pdo->prepare("
            INSERT INTO activity_logs 
            (user_id, action, details, ip_address, created_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $details = json_encode([
            'type' => 'validation_dechet',
            'dechet_id' => $dechet_id,
            'ancien_statut' => $dechet['statut'],
            'nouveau_statut' => $nouveau_statut,
            'site_nom' => $dechet['nom_site'],
            'referent' => trim($dechet['referent_prenom'] . ' ' . $dechet['referent_nom']),
            'commentaire' => $commentaire
        ]);
        
        $stmt->execute([
            $user['id'],
            "Validation relevé déchets #{$dechet_id}",
            $details,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ]);

        // Valider la transaction
        $pdo->commit();

        // Récupérer les données mises à jour
        $stmt = $pdo->prepare("
            SELECT dcr.*, s.nom_site, s.region, 
                   u.nom as referent_nom, u.prenom as referent_prenom,
                   v.nom as validateur_nom, v.prenom as validateur_prenom
            FROM dechets_collecte_releves dcr
            LEFT JOIN sites s ON dcr.site_id = s.id  
            LEFT JOIN users u ON dcr.utilisateur_id = u.id
            LEFT JOIN users v ON dcr.validateur_id = v.id
            WHERE dcr.id = ?
        ");
        $stmt->execute([$dechet_id]);
        $dechet_updated = $stmt->fetch(PDO::FETCH_ASSOC);

        $response = [
            'success' => true,
            'message' => "Relevé de déchets #{$dechet_id} " . ($nouveau_statut === 'valide' ? 'validé' : 'rejeté') . " avec succès",
            'data' => [
                'id' => $dechet_id,
                'ancien_statut' => $dechet['statut'],
                'nouveau_statut' => $nouveau_statut,
                'validateur' => [
                    'id' => $user['id'],
                    'nom' => trim($user['prenom'] . ' ' . $user['nom']),
                    'role' => $user['role']
                ],
                'site' => [
                    'id' => $dechet['site_id'],
                    'nom' => $dechet['nom_site']
                ],
                'referent' => trim($dechet['referent_prenom'] . ' ' . $dechet['referent_nom']),
                'date_validation' => $dechet_updated['date_validation'],
                'commentaire' => $commentaire
            ]
        ];

        echo json_encode($response, JSON_UNESCAPED_UNICODE);

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }

} catch (PDOException $e) {
    error_log("Erreur validation déchets: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur base de données lors de la validation',
        'error_details' => $e->getMessage()
    ]);
} catch (Exception $e) {
    error_log("Erreur validation déchets: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Erreur serveur lors de la validation',
        'error_details' => $e->getMessage()
    ]);
}
?>