<?php
// api/export/global.php - VERSION SANS VÉRIFICATION TOKEN
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';

// UTILISATEUR PAR DÉFAUT (SANS AUTHENTIFICATION)
$user = [
    'nom' => 'Système',
    'prenom' => 'Export',
    'role' => 'admin'
];

try {
    $database = new Database();
    $db = $database->getConnection();
    
    // Paramètres d'export
    $format = $_GET['format'] ?? 'excel_multiple';
    $types = $_GET['types'] ?? ['profils', 'dechets', 'sites'];
    $include_relations = $_GET['include_relations'] ?? 'true';
    $include_stats = $_GET['include_stats'] ?? 'true';
    
    if (!is_array($types)) {
        $types = [$types];
    }
    
    $export_data = [
        'metadata' => [
            'export_date' => date('Y-m-d H:i:s'),
            'exported_by' => $user['nom'] . ' ' . $user['prenom'],
            'export_type' => 'global',
            'format' => $format,
            'types_included' => $types,
            'include_relations' => $include_relations === 'true',
            'include_stats' => $include_stats === 'true'
        ]
    ];
    
    // Export des profils
    if (in_array('profils', $types)) {
        $export_data['profils'] = exportProfils($db);
        if ($include_relations === 'true') {
            $export_data['evenements_extremes_profils'] = exportEvenementsExtremesP($db);
            $export_data['mesures_topographiques'] = exportMesuresTopographiques($db);
        }
    }
    
    // Export des déchets
    if (in_array('dechets', $types)) {
        $export_data['releves_dechets'] = exportDechets($db);
        if ($include_relations === 'true') {
            $export_data['dechets_detail'] = exportDechetsDetail($db);
            $export_data['evenements_extremes_dechets'] = exportEvenementsExtremesD($db);
        }
    }
    
    // Export des sites
    if (in_array('sites', $types)) {
        $export_data['sites'] = exportSites($db);
    }
    
    // Export des utilisateurs (toujours autorisé)
    if (in_array('users', $types)) {
        $export_data['users'] = exportUsers($db);
    }
    
    // Générer les statistiques
    if ($include_stats === 'true') {
        $export_data['statistiques_globales'] = generateGlobalStats($db, $types);
    }
    
    // Générer le fichier selon le format
    switch ($format) {
        case 'excel_multiple':
            generateGlobalExcelExport($export_data);
            break;
        case 'zip_csv':
            generateZipCSVExport($export_data);
            break;
        case 'json':
            header('Content-Type: application/json; charset=utf-8');
            header('Content-Disposition: attachment; filename="export_global_' . date('Y-m-d_H-i-s') . '.json"');
            echo json_encode($export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            exit;
        default:
            throw new Exception('Format d\'export non supporté');
    }
    
} catch (Exception $e) {
    error_log("Erreur export global: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur lors de l\'export global: ' . $e->getMessage()
    ]);
}

/**
 * Exporte les profils topographiques
 */
function exportProfils($db) {
    $query = "
        SELECT 
            pt.*,
            CONCAT(u.prenom, ' ', u.nom) as referent_nom,
            u.email as referent_email,
            u.telephone as referent_telephone,
            s.nom_site,
            s.region,
            s.latitude as site_latitude,
            s.longitude as site_longitude
        FROM profils_topographiques pt
        LEFT JOIN users u ON pt.user_id = u.id
        LEFT JOIN sites s ON pt.site_id = s.id
        ORDER BY pt.date_releve DESC
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les relevés de déchets
 */
function exportDechets($db) {
    $query = "
        SELECT 
            rd.*,
            CONCAT(u.prenom, ' ', u.nom) as referent_nom,
            u.email as referent_email,
            u.telephone as referent_telephone,
            s.nom_site,
            s.region,
            s.latitude as site_latitude,
            s.longitude as site_longitude
        FROM releves_dechets rd
        LEFT JOIN users u ON rd.user_id = u.id
        LEFT JOIN sites s ON rd.site_id = s.id
        ORDER BY rd.date_releve DESC
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les sites
 */
function exportSites($db) {
    $query = "
        SELECT 
            s.*,
            COUNT(DISTINCT pt.id) as nb_profils,
            COUNT(DISTINCT rd.id) as nb_releves_dechets,
            COUNT(DISTINCT u.id) as nb_referents
        FROM sites s
        LEFT JOIN profils_topographiques pt ON s.id = pt.site_id
        LEFT JOIN releves_dechets rd ON s.id = rd.site_id
        LEFT JOIN users u ON s.id = u.site_id AND u.role = 'referent'
        GROUP BY s.id
        ORDER BY s.nom_site
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les utilisateurs
 */
function exportUsers($db) {
    $query = "
        SELECT 
            u.id,
            u.nom,
            u.prenom,
            u.email,
            u.telephone,
            u.role,
            u.actif,
            u.created_at,
            u.updated_at,
            s.nom_site,
            s.region,
            CASE WHEN u.photo IS NOT NULL AND LENGTH(u.photo) > 0 THEN 1 ELSE 0 END as has_photo
        FROM users u
        LEFT JOIN sites s ON u.site_id = s.id
        ORDER BY u.nom, u.prenom
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les événements extrêmes des profils
 */
function exportEvenementsExtremesP($db) {
    $query = "
        SELECT 
            eep.*,
            pt.code_referent,
            pt.date_releve
        FROM evenements_extremes_profils eep
        LEFT JOIN profils_topographiques pt ON eep.profil_id = pt.id
        ORDER BY pt.date_releve DESC
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les mesures topographiques
 */
function exportMesuresTopographiques($db) {
    $query = "
        SELECT 
            mt.*,
            pt.code_referent,
            pt.date_releve
        FROM mesures_topographiques mt
        LEFT JOIN profils_topographiques pt ON mt.profil_id = pt.id
        ORDER BY pt.date_releve DESC, mt.numero_ordre
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les détails des déchets
 */
function exportDechetsDetail($db) {
    $query = "
        SELECT 
            dd.id,
            dd.releve_id,
            dd.categorie,
            dd.nombre_articles,
            dd.poids,
            dd.source_pollution,
            dd.created_at,
            CASE WHEN dd.photo IS NOT NULL AND LENGTH(dd.photo) > 0 THEN 1 ELSE 0 END as has_photo,
            rd.code_referent,
            rd.date_releve
        FROM dechets_detail dd
        LEFT JOIN releves_dechets rd ON dd.releve_id = rd.id
        ORDER BY rd.date_releve DESC, dd.id
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Exporte les événements extrêmes des déchets
 */
function exportEvenementsExtremesD($db) {
    $query = "
        SELECT 
            eed.*,
            rd.code_referent,
            rd.date_releve
        FROM evenements_extremes_dechets eed
        LEFT JOIN releves_dechets rd ON eed.releve_id = rd.id
        ORDER BY rd.date_releve DESC
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Génère les statistiques globales
 */
function generateGlobalStats($db, $types) {
    $stats = [
        'resume_executif' => [],
        'par_site' => [],
        'par_mois' => [],
        'par_categorie_dechets' => []
    ];
    
    // Résumé exécutif
    if (in_array('profils', $types)) {
        $query = "SELECT COUNT(*) as total FROM profils_topographiques";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $stats['resume_executif']['total_profils'] = $stmt->fetchColumn();
        
        $query = "SELECT COUNT(*) as total FROM mesures_topographiques";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $stats['resume_executif']['total_mesures'] = $stmt->fetchColumn();
    }
    
    if (in_array('dechets', $types)) {
        $query = "SELECT COUNT(*) as total FROM releves_dechets";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $stats['resume_executif']['total_releves_dechets'] = $stmt->fetchColumn();
        
        $query = "SELECT SUM(poids) as total FROM dechets_detail";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $stats['resume_executif']['total_poids_dechets'] = round($stmt->fetchColumn() ?: 0, 2);
    }
    
    if (in_array('sites', $types)) {
        $query = "SELECT COUNT(*) as total FROM sites";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $stats['resume_executif']['total_sites'] = $stmt->fetchColumn();
    }
    
    // Statistiques par site
    $query = "
        SELECT 
            s.nom_site,
            s.region,
            COUNT(DISTINCT pt.id) as nb_profils,
            COUNT(DISTINCT rd.id) as nb_releves_dechets,
            COALESCE(SUM(dd.poids), 0) as total_poids_dechets
        FROM sites s
        LEFT JOIN profils_topographiques pt ON s.id = pt.site_id
        LEFT JOIN releves_dechets rd ON s.id = rd.site_id
        LEFT JOIN dechets_detail dd ON rd.id = dd.releve_id
        GROUP BY s.id, s.nom_site, s.region
        ORDER BY nb_profils + nb_releves_dechets DESC
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $stats['par_site'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Statistiques par mois (12 derniers mois)
    $query = "
        SELECT 
            DATE_FORMAT(date_releve, '%Y-%m') as mois,
            DATE_FORMAT(date_releve, '%M %Y') as mois_nom,
            COUNT(DISTINCT pt.id) as nb_profils,
            COUNT(DISTINCT rd.id) as nb_releves_dechets
        FROM (
            SELECT date_releve FROM profils_topographiques 
            WHERE date_releve >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
            UNION ALL
            SELECT date_releve FROM releves_dechets 
            WHERE date_releve >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
        ) all_dates
        LEFT JOIN profils_topographiques pt ON all_dates.date_releve = pt.date_releve
        LEFT JOIN releves_dechets rd ON all_dates.date_releve = rd.date_releve
        GROUP BY mois, mois_nom
        ORDER BY mois DESC
        LIMIT 12
    ";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $stats['par_mois'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Statistiques par catégorie de déchets
    if (in_array('dechets', $types)) {
        $categories = [
            'PL' => 'Plastique',
            'CA' => 'Carton/Papier', 
            'TX' => 'Textile',
            'ME' => 'Métal',
            'VE' => 'Verre',
            'BO' => 'Bois',
            'AU' => 'Autre'
        ];
        
        $query = "
            SELECT 
                dd.categorie,
                COUNT(*) as nb_releves,
                SUM(dd.nombre_articles) as total_articles,
                SUM(dd.poids) as total_poids,
                AVG(dd.poids) as poids_moyen
            FROM dechets_detail dd
            GROUP BY dd.categorie
            ORDER BY total_poids DESC
        ";
        
        $stmt = $db->prepare($query);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($results as $result) {
            $stats['par_categorie_dechets'][] = [
                'categorie' => $result['categorie'],
                'categorie_nom' => $categories[$result['categorie']] ?? 'Inconnu',
                'nb_releves' => $result['nb_releves'],
                'total_articles' => $result['total_articles'],
                'total_poids' => round($result['total_poids'], 2),
                'poids_moyen' => round($result['poids_moyen'], 2)
            ];
        }
    }
    
    return $stats;
}

/**
 * Génère un export Excel global avec multiples feuilles
 */
function generateGlobalExcelExport($data) {
    $vendorPath = __DIR__ . '/../vendor/autoload.php';
    
    if (!file_exists($vendorPath)) {
        // Fallback vers JSON si PhpSpreadsheet non disponible
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="export_global_' . date('Y-m-d_H-i-s') . '.json"');
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    require_once $vendorPath;
    
    use PhpOffice\PhpSpreadsheet\Spreadsheet;
    use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
    use PhpOffice\PhpSpreadsheet\Style\Alignment;
    use PhpOffice\PhpSpreadsheet\Style\Border;
    use PhpOffice\PhpSpreadsheet\Style\Fill;
    
    $spreadsheet = new Spreadsheet();
    
    // Feuille 1: Métadonnées et résumé
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setTitle('Résumé Exécutif');
    
    // Titre principal
    $sheet->setCellValue('A1', 'Export Global des Données Côtières');
    $sheet->mergeCells('A1:E1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    // Informations d'export
    $row = 3;
    $sheet->setCellValue('A' . $row, 'Date d\'export:');
    $sheet->setCellValue('B' . $row, $data['metadata']['export_date']);
    $row++;
    $sheet->setCellValue('A' . $row, 'Exporté par:');
    $sheet->setCellValue('B' . $row, $data['metadata']['exported_by']);
    $row++;
    $sheet->setCellValue('A' . $row, 'Types inclus:');
    $sheet->setCellValue('B' . $row, implode(', ', $data['metadata']['types_included']));
    
    // Statistiques globales
    if (isset($data['statistiques_globales']['resume_executif'])) {
        $row += 2;
        $sheet->setCellValue('A' . $row, 'STATISTIQUES GLOBALES');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(14);
        $row++;
        
        $stats = $data['statistiques_globales']['resume_executif'];
        foreach ($stats as $key => $value) {
            $label = str_replace('_', ' ', ucwords($key, '_'));
            $sheet->setCellValue('A' . $row, $label . ':');
            $sheet->setCellValue('B' . $row, $value);
            $row++;
        }
    }
    
    $sheetIndex = 0;
    
    // Feuille pour les profils
    if (isset($data['profils'])) {
        $profilSheet = $spreadsheet->createSheet();
        $profilSheet->setTitle('Profils Topographiques');
        createProfilSheet($profilSheet, $data['profils']);
    }
    
    // Feuille pour les déchets
    if (isset($data['releves_dechets'])) {
        $dechetSheet = $spreadsheet->createSheet();
        $dechetSheet->setTitle('Relevés Déchets');
        createDechetSheet($dechetSheet, $data['releves_dechets']);
    }
    
    // Feuille pour les sites
    if (isset($data['sites'])) {
        $siteSheet = $spreadsheet->createSheet();
        $siteSheet->setTitle('Sites');
        createSiteSheet($siteSheet, $data['sites']);
    }
    
    // Feuille pour les utilisateurs
    if (isset($data['users'])) {
        $userSheet = $spreadsheet->createSheet();
        $userSheet->setTitle('Utilisateurs');
        createUserSheet($userSheet, $data['users']);
    }
    
    // Feuilles pour les données de relation
    if (isset($data['mesures_topographiques'])) {
        $mesureSheet = $spreadsheet->createSheet();
        $mesureSheet->setTitle('Mesures Topographiques');
        createMesureSheet($mesureSheet, $data['mesures_topographiques']);
    }
    
    if (isset($data['dechets_detail'])) {
        $detailSheet = $spreadsheet->createSheet();
        $detailSheet->setTitle('Détails Déchets');
        createDechetDetailSheet($detailSheet, $data['dechets_detail']);
    }
    
    // Feuille de statistiques
    if (isset($data['statistiques_globales'])) {
        $statsSheet = $spreadsheet->createSheet();
        $statsSheet->setTitle('Statistiques');
        createStatsSheet($statsSheet, $data['statistiques_globales']);
    }
    
    // Auto-ajuster toutes les colonnes
    foreach ($spreadsheet->getAllSheets() as $worksheet) {
        $worksheet->calculateColumnWidths();
        foreach (range('A', $worksheet->getHighestColumn()) as $col) {
            $worksheet->getColumnDimension($col)->setAutoSize(true);
        }
    }
    
    // Sélectionner la première feuille
    $spreadsheet->setActiveSheetIndex(0);
    
    // Générer le fichier
    $filename = 'export_global_' . date('Y-m-d_H-i-s') . '.xlsx';
    
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');
    
    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');
    exit;
}

/**
 * Génère un export ZIP avec plusieurs fichiers CSV
 */
function generateZipCSVExport($data) {
    $zipFile = 'export_global_' . date('Y-m-d_H-i-s') . '.zip';
    
    // Créer un fichier temporaire pour le ZIP
    $tempZip = tempnam(sys_get_temp_dir(), 'export_zip');
    
    $zip = new ZipArchive();
    if ($zip->open($tempZip, ZipArchive::CREATE) !== TRUE) {
        throw new Exception('Impossible de créer le fichier ZIP');
    }
    
    // Ajouter chaque type de données comme fichier CSV
    foreach ($data as $key => $items) {
        if ($key === 'metadata' || $key === 'statistiques_globales' || empty($items)) {
            continue;
        }
        
        $csvContent = generateCSVContent($items, $key);
        $zip->addFromString($key . '.csv', $csvContent);
    }
    
    // Ajouter un fichier de métadonnées
    $metadataContent = "Export Global des Données\n";
    $metadataContent .= "Date: " . $data['metadata']['export_date'] . "\n";
    $metadataContent .= "Exporté par: " . $data['metadata']['exported_by'] . "\n";
    $metadataContent .= "Types: " . implode(', ', $data['metadata']['types_included']) . "\n";
    
    $zip->addFromString('metadata.txt', $metadataContent);
    
    $zip->close();
    
    // Envoyer le fichier
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $zipFile . '"');
    header('Cache-Control: max-age=0');
    
    readfile($tempZip);
    unlink($tempZip);
    exit;
}

/**
 * Génère le contenu CSV pour un ensemble de données
 */
function generateCSVContent($data, $type) {
    if (empty($data)) {
        return '';
    }
    
    $output = fopen('php://temp', 'w');
    
    // BOM UTF-8
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // En-têtes (utiliser les clés du premier élément)
    $headers = array_keys($data[0]);
    fputcsv($output, $headers, ';');
    
    // Données
    foreach ($data as $row) {
        fputcsv($output, $row, ';');
    }
    
    rewind($output);
    $csvContent = stream_get_contents($output);
    fclose($output);
    
    return $csvContent;
}

/**
 * Fonctions de création des feuilles Excel
 */
function createProfilSheet($sheet, $profils) {
    $headers = [
        'ID', 'Code Référent', 'N° Profil', 'Date Relevé', 'Heure',
        'Latitude', 'Longitude', 'Altitude', 'Marée', 'Statut',
        'Référent', 'Site', 'Région', 'Créé le'
    ];
    
    // En-têtes
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    // Style des en-têtes
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '4472C4']
        ]
    ]);
    
    // Données
    $row = 2;
    foreach ($profils as $profil) {
        $sheet->setCellValue('A' . $row, $profil['id']);
        $sheet->setCellValue('B' . $row, $profil['code_referent'] ?? '');
        $sheet->setCellValue('C' . $row, $profil['numero_profil_borne'] ?? '');
        $sheet->setCellValue('D' . $row, $profil['date_releve'] ?? '');
        $sheet->setCellValue('E' . $row, $profil['heure_debut'] ?? '');
        $sheet->setCellValue('F' . $row, $profil['latitude_borne'] ?? '');
        $sheet->setCellValue('G' . $row, $profil['longitude_borne'] ?? '');
        $sheet->setCellValue('H' . $row, $profil['altitude_borne'] ?? '');
        $sheet->setCellValue('I' . $row, $profil['maree_type'] ?? '');
        $sheet->setCellValue('J' . $row, $profil['statut'] ?? '');
        $sheet->setCellValue('K' . $row, $profil['referent_nom'] ?? '');
        $sheet->setCellValue('L' . $row, $profil['nom_site'] ?? '');
        $sheet->setCellValue('M' . $row, $profil['region'] ?? '');
        $sheet->setCellValue('N' . $row, $profil['created_at'] ?? '');
        $row++;
    }
}

function createDechetSheet($sheet, $dechets) {
    $headers = [
        'ID', 'Code Référent', 'Date Relevé', 'Heure',
        'Lat. Début', 'Long. Début', 'Lat. Fin', 'Long. Fin',
        'Statut', 'Référent', 'Site', 'Région', 'Créé le'
    ];
    
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => 'F39C12']
        ]
    ]);
    
    $row = 2;
    foreach ($dechets as $dechet) {
        $sheet->setCellValue('A' . $row, $dechet['id']);
        $sheet->setCellValue('B' . $row, $dechet['code_referent'] ?? '');
        $sheet->setCellValue('C' . $row, $dechet['date_releve'] ?? '');
        $sheet->setCellValue('D' . $row, $dechet['heure_debut'] ?? '');
        $sheet->setCellValue('E' . $row, $dechet['latitude_debut'] ?? '');
        $sheet->setCellValue('F' . $row, $dechet['longitude_debut'] ?? '');
        $sheet->setCellValue('G' . $row, $dechet['latitude_fin'] ?? '');
        $sheet->setCellValue('H' . $row, $dechet['longitude_fin'] ?? '');
        $sheet->setCellValue('I' . $row, $dechet['statut'] ?? '');
        $sheet->setCellValue('J' . $row, $dechet['referent_nom'] ?? '');
        $sheet->setCellValue('K' . $row, $dechet['nom_site'] ?? '');
        $sheet->setCellValue('L' . $row, $dechet['region'] ?? '');
        $sheet->setCellValue('M' . $row, $dechet['created_at'] ?? '');
        $row++;
    }
}

function createSiteSheet($sheet, $sites) {
    $headers = [
        'ID', 'Nom Site', 'Région', 'Latitude', 'Longitude',
        'Nb Profils', 'Nb Relevés Déchets', 'Nb Référents', 'Créé le'
    ];
    
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '27AE60']
        ]
    ]);
    
    $row = 2;
    foreach ($sites as $site) {
        $sheet->setCellValue('A' . $row, $site['id']);
        $sheet->setCellValue('B' . $row, $site['nom_site'] ?? '');
        $sheet->setCellValue('C' . $row, $site['region'] ?? '');
        $sheet->setCellValue('D' . $row, $site['latitude'] ?? '');
        $sheet->setCellValue('E' . $row, $site['longitude'] ?? '');
        $sheet->setCellValue('F' . $row, $site['nb_profils'] ?? 0);
        $sheet->setCellValue('G' . $row, $site['nb_releves_dechets'] ?? 0);
        $sheet->setCellValue('H' . $row, $site['nb_referents'] ?? 0);
        $sheet->setCellValue('I' . $row, $site['created_at'] ?? '');
        $row++;
    }
}

function createUserSheet($sheet, $users) {
    $headers = [
        'ID', 'Nom', 'Prénom', 'Email', 'Téléphone', 'Rôle',
        'Site', 'Région', 'Actif', 'Photo', 'Créé le'
    ];
    
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '8E44AD']
        ]
    ]);
    
    $row = 2;
    foreach ($users as $user) {
        $sheet->setCellValue('A' . $row, $user['id']);
        $sheet->setCellValue('B' . $row, $user['nom'] ?? '');
        $sheet->setCellValue('C' . $row, $user['prenom'] ?? '');
        $sheet->setCellValue('D' . $row, $user['email'] ?? '');
        $sheet->setCellValue('E' . $row, $user['telephone'] ?? '');
        $sheet->setCellValue('F' . $row, $user['role'] ?? '');
        $sheet->setCellValue('G' . $row, $user['nom_site'] ?? '');
        $sheet->setCellValue('H' . $row, $user['region'] ?? '');
        $sheet->setCellValue('I' . $row, $user['actif'] ? 'Oui' : 'Non');
        $sheet->setCellValue('J' . $row, $user['has_photo'] ? 'Oui' : 'Non');
        $sheet->setCellValue('K' . $row, $user['created_at'] ?? '');
        $row++;
    }
}

function createMesureSheet($sheet, $mesures) {
    $headers = [
        'ID', 'Profil ID', 'Code Référent', 'N° Ordre', 'Distance (m)',
        'Dénivelé (cm)', 'Dénivelé Cumulé (cm)', 'Altitude (m)', 'Commentaires'
    ];
    
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '17A2B8']
        ]
    ]);
    
    $row = 2;
    foreach ($mesures as $mesure) {
        $sheet->setCellValue('A' . $row, $mesure['id'] ?? '');
        $sheet->setCellValue('B' . $row, $mesure['profil_id'] ?? '');
        $sheet->setCellValue('C' . $row, $mesure['code_referent'] ?? '');
        $sheet->setCellValue('D' . $row, $mesure['numero_ordre'] ?? '');
        $sheet->setCellValue('E' . $row, $mesure['distance'] ?? '');
        $sheet->setCellValue('F' . $row, $mesure['denivele'] ?? '');
        $sheet->setCellValue('G' . $row, $mesure['denivele_cumule'] ?? '');
        $sheet->setCellValue('H' . $row, $mesure['altitude'] ?? '');
        $sheet->setCellValue('I' . $row, $mesure['commentaires'] ?? '');
        $row++;
    }
}

function createDechetDetailSheet($sheet, $details) {
    $headers = [
        'ID', 'Relevé ID', 'Code Référent', 'Catégorie', 'Articles',
        'Poids (kg)', 'Source Pollution', 'Photo', 'Créé le'
    ];
    
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '1', $header);
        $col++;
    }
    
    $sheet->getStyle('A1:' . chr(ord('A') + count($headers) - 1) . '1')->applyFromArray([
        'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => 'FD7E14']
        ]
    ]);
    
    $row = 2;
    foreach ($details as $detail) {
        $sheet->setCellValue('A' . $row, $detail['id'] ?? '');
        $sheet->setCellValue('B' . $row, $detail['releve_id'] ?? '');
        $sheet->setCellValue('C' . $row, $detail['code_referent'] ?? '');
        $sheet->setCellValue('D' . $row, $detail['categorie'] ?? '');
        $sheet->setCellValue('E' . $row, $detail['nombre_articles'] ?? '');
        $sheet->setCellValue('F' . $row, $detail['poids'] ?? '');
        $sheet->setCellValue('G' . $row, $detail['source_pollution'] ?? '');
        $sheet->setCellValue('H' . $row, $detail['has_photo'] ? 'Oui' : 'Non');
        $sheet->setCellValue('I' . $row, $detail['created_at'] ?? '');
        $row++;
    }
}

function createStatsSheet($sheet, $stats) {
    $row = 1;
    
    // Résumé exécutif
    $sheet->setCellValue('A' . $row, 'RÉSUMÉ EXÉCUTIF');
    $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(14);
    $row += 2;
    
    if (isset($stats['resume_executif'])) {
        foreach ($stats['resume_executif'] as $key => $value) {
            $label = str_replace('_', ' ', ucwords($key, '_'));
            $sheet->setCellValue('A' . $row, $label . ':');
            $sheet->setCellValue('B' . $row, $value);
            $row++;
        }
    }
    
    // Statistiques par site
    if (isset($stats['par_site']) && !empty($stats['par_site'])) {
        $row += 2;
        $sheet->setCellValue('A' . $row, 'STATISTIQUES PAR SITE');
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(14);
        $row++;
        
        // En-têtes
        $sheet->setCellValue('A' . $row, 'Site');
        $sheet->setCellValue('B' . $row, 'Région');
        $sheet->setCellValue('C' . $row, 'Nb Profils');
        $sheet->setCellValue('D' . $row, 'Nb Relevés Déchets');
        $sheet->setCellValue('E' . $row, 'Total Poids Déchets');
        $row++;
        
        foreach ($stats['par_site'] as $site) {
            $sheet->setCellValue('A' . $row, $site['nom_site']);
            $sheet->setCellValue('B' . $row, $site['region']);
            $sheet->setCellValue('C' . $row, $site['nb_profils']);
            $sheet->setCellValue('D' . $row, $site['nb_releves_dechets']);
            $sheet->setCellValue('E' . $row, $site['total_poids_dechets']);
            $row++;
        }
    }
}
?>