<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Gérer les requêtes OPTIONS pour CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    // Vérifier que c'est une requête POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Méthode non autorisée');
    }
    
    // Récupérer les données JSON
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Données JSON invalides');
    }
    
    // Validation des champs obligatoires
    $required_fields = ['nom_site', 'region', 'latitude', 'longitude'];
    foreach ($required_fields as $field) {
        if (empty($input[$field]) && $input[$field] !== 0) {
            throw new Exception("Le champ '$field' est obligatoire");
        }
    }
    
    // Validation des coordonnées
    $latitude = (float) $input['latitude'];
    $longitude = (float) $input['longitude'];
    
    if ($latitude < -90 || $latitude > 90) {
        throw new Exception('La latitude doit être comprise entre -90 et 90');
    }
    
    if ($longitude < -180 || $longitude > 180) {
        throw new Exception('La longitude doit être comprise entre -180 et 180');
    }
    
    // Utiliser votre classe Database
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Impossible de se connecter à la base de données');
    }
    
    // Vérifier si le nom du site existe déjà
    $checkSql = "SELECT id FROM sites WHERE nom_site = ? AND region = ?";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$input['nom_site'], $input['region']]);
    
    if ($checkStmt->fetch()) {
        throw new Exception('Un site avec ce nom existe déjà dans cette région');
    }
    
    // Préparer les données
    $nom_site = trim($input['nom_site']);
    $region = trim($input['region']);
    
    // Insertion dans la base de données
    $sql = "INSERT INTO sites (nom_site, region, latitude, longitude, created_at) 
            VALUES (?, ?, ?, ?, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        $nom_site, 
        $region, 
        $latitude, 
        $longitude
    ]);
    
    $siteId = $pdo->lastInsertId();
    
    // Récupérer le site créé
    $getSiteSql = "SELECT s.id, s.nom_site, s.region, s.latitude, s.longitude, s.created_at,
                          0 as nb_utilisateurs, 0 as nb_profils_topo, 0 as nb_releves_dechets
                   FROM sites s WHERE s.id = ?";
    $getSiteStmt = $pdo->prepare($getSiteSql);
    $getSiteStmt->execute([$siteId]);
    $newSite = $getSiteStmt->fetch(PDO::FETCH_ASSOC);
    
    // Formater la réponse
    if ($newSite) {
        $newSite['latitude'] = (float) $newSite['latitude'];
        $newSite['longitude'] = (float) $newSite['longitude'];
        $newSite['coordonnees_display'] = number_format($newSite['latitude'], 6) . ', ' . number_format($newSite['longitude'], 6);
        $newSite['stats'] = [
            'utilisateurs' => 0,
            'profils_topographiques' => 0,
            'releves_dechets' => 0,
            'total_activites' => 0
        ];
        unset($newSite['nb_utilisateurs'], $newSite['nb_profils_topo'], $newSite['nb_releves_dechets']);
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Site créé avec succès',
        'data' => $newSite,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur de base de données: ' . $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>