<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    // Connexion à la base de données
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Erreur de connexion à la base de données');
    }
    
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Récupérer les données pour synchronisation descendante (serveur -> mobile)
        handleDownSync($pdo);
    } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Recevoir les données pour synchronisation ascendante (mobile -> serveur)
        handleUpSync($pdo);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

function handleDownSync($pdo) {
    // Récupérer la dernière date de sync depuis les paramètres GET
    $lastSync = isset($_GET['last_sync']) ? $_GET['last_sync'] : '2000-01-01 00:00:00';
    
    // Récupérer les sites
    $sitesStmt = $pdo->query("SELECT * FROM sites ORDER BY nom_site");
    $sites = $sitesStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Récupérer les profils topographiques récents
    $profilsStmt = $pdo->prepare("
        SELECT pt.*, s.nom_site, s.region, u.nom as referent_nom, u.prenom as referent_prenom
        FROM profils_topographiques pt
        LEFT JOIN sites s ON pt.site_id = s.id
        LEFT JOIN utilisateurs u ON pt.utilisateur_id = u.id
        WHERE pt.created_at > ?
        ORDER BY pt.created_at DESC
        LIMIT 100
    ");
    $profilsStmt->execute([$lastSync]);
    $profils = $profilsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Récupérer les relevés de déchets récents
    $dechetsStmt = $pdo->prepare("
        SELECT rd.*, s.nom_site, s.region, u.nom as referent_nom, u.prenom as referent_prenom
        FROM releves_dechets rd
        LEFT JOIN sites s ON rd.site_id = s.id
        LEFT JOIN utilisateurs u ON rd.utilisateur_id = u.id
        WHERE rd.created_at > ?
        ORDER BY rd.created_at DESC
        LIMIT 100
    ");
    $dechetsStmt->execute([$lastSync]);
    $dechets = $dechetsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Récupérer les utilisateurs actifs
    $usersStmt = $pdo->query("
        SELECT u.id, u.nom, u.prenom, u.email, u.role, s.nom_site, s.region
        FROM utilisateurs u
        LEFT JOIN sites s ON u.site_id = s.id
        WHERE u.actif = 1
        ORDER BY u.nom
    ");
    $users = $usersStmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'message' => 'Synchronisation descendante réussie',
        'data' => [
            'sites' => $sites,
            'profils_topographiques' => $profils,
            'releves_dechets' => $dechets,
            'utilisateurs' => $users,
            'sync_time' => date('Y-m-d H:i:s'),
            'last_sync_requested' => $lastSync
        ],
        'counts' => [
            'sites' => count($sites),
            'profils' => count($profils),
            'dechets' => count($dechets),
            'users' => count($users)
        ]
    ]);
}

function handleUpSync($pdo) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Données JSON invalides');
    }
    
    $results = [
        'profils_created' => 0,
        'dechets_created' => 0,
        'errors' => []
    ];
    
    // Traiter les profils topographiques
    if (isset($input['profils']) && is_array($input['profils'])) {
        foreach ($input['profils'] as $profil) {
            try {
                $sql = "INSERT INTO profils_topographiques 
                        (code_referent, utilisateur_id, site_id, date_releve, heure_debut, 
                         latitude_borne, longitude_borne, altitude_borne, azimut, maree, 
                         evenements_extremes, statut, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'soumis', NOW())";
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $profil['code_referent'] ?? '',
                    $profil['utilisateur_id'] ?? null,
                    $profil['site_id'] ?? null,
                    $profil['date_releve'] ?? date('Y-m-d'),
                    $profil['heure_debut'] ?? date('H:i:s'),
                    $profil['latitude_borne'] ?? null,
                    $profil['longitude_borne'] ?? null,
                    $profil['altitude_borne'] ?? null,
                    $profil['azimut'] ?? null,
                    $profil['maree'] ?? null,
                    $profil['evenements_extremes'] ?? null
                ]);
                
                $results['profils_created']++;
                
            } catch (Exception $e) {
                $results['errors'][] = "Erreur profil: " . $e->getMessage();
            }
        }
    }
    
    // Traiter les relevés de déchets
    if (isset($input['dechets']) && is_array($input['dechets'])) {
        foreach ($input['dechets'] as $dechet) {
            try {
                $sql = "INSERT INTO releves_dechets 
                        (code_referent, utilisateur_id, site_id, date_releve, heure_debut, 
                         latitude_debut, longitude_debut, orientation_plage, evenements_extremes, 
                         statut, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'soumis', NOW())";
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $dechet['code_referent'] ?? '',
                    $dechet['utilisateur_id'] ?? null,
                    $dechet['site_id'] ?? null,
                    $dechet['date_releve'] ?? date('Y-m-d'),
                    $dechet['heure_debut'] ?? date('H:i:s'),
                    $dechet['latitude_debut'] ?? null,
                    $dechet['longitude_debut'] ?? null,
                    $dechet['orientation_plage'] ?? null,
                    $dechet['evenements_extremes'] ?? null
                ]);
                
                $results['dechets_created']++;
                
            } catch (Exception $e) {
                $results['errors'][] = "Erreur déchet: " . $e->getMessage();
            }
        }
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Synchronisation ascendante réussie',
        'data' => $results,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>