<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, Accept');

// Gérer les requêtes OPTIONS pour CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    // Vérifier que c'est une requête POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Méthode non autorisée');
    }
    
    // Récupérer les données JSON
    $rawInput = file_get_contents('php://input');
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        throw new Exception('Données JSON invalides: ' . json_last_error_msg());
    }
    
    // Log pour debug
    error_log("Données reçues: " . print_r($input, true));
    
    // Validation des champs obligatoires
    $required_fields = ['nom', 'email', 'mot_de_passe'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            throw new Exception("Le champ '$field' est obligatoire");
        }
    }
    
    // Validation de l'email
    if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Format d\'email invalide');
    }
    
    // Utiliser votre classe Database
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Impossible de se connecter à la base de données');
    }
    
    // Vérifier si l'email existe déjà
    $checkSql = "SELECT id FROM utilisateurs WHERE email = ?";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$input['email']]);
    
    if ($checkStmt->fetch()) {
        throw new Exception('Un utilisateur avec cet email existe déjà');
    }
    
    // Préparer les données
    $nom = trim($input['nom']);
    $prenom = isset($input['prenom']) ? trim($input['prenom']) : '';
    $email = trim(strtolower($input['email']));
    
    // Gérer le mot de passe (peut venir de 'password' ou 'mot_de_passe')
    $password_raw = '';
    if (isset($input['mot_de_passe']) && !empty($input['mot_de_passe'])) {
        $password_raw = $input['mot_de_passe'];
    } elseif (isset($input['password']) && !empty($input['password'])) {
        $password_raw = $input['password'];
    } else {
        throw new Exception('Le mot de passe est obligatoire');
    }
    
    $password = md5($password_raw); // Utiliser MD5 comme dans votre structure existante
    $role = isset($input['role']) ? $input['role'] : 'referent';
    $telephone = isset($input['telephone']) ? trim($input['telephone']) : null;
    $site_id = null;
    
    // Gestion du site_id
    if (isset($input['site_id']) && !empty($input['site_id']) && $input['site_id'] !== 'null' && $input['site_id'] !== '') {
        $site_id = intval($input['site_id']);
    }
    
    $actif = isset($input['actif']) ? intval($input['actif']) : 1; // Actif par défaut
    
    // Validation du rôle
    $roles_autorises = ['controleur', 'referent', 'superviseur', 'admin'];
    if (!in_array($role, $roles_autorises)) {
        $role = 'referent';
    }
    
    // Log pour debug
    error_log("Données préparées - Nom: $nom, Email: $email, Role: $role, Site_ID: $site_id");
    
    // Insertion dans la base de données
    $sql = "INSERT INTO utilisateurs (nom, prenom, email, mot_de_passe, role, telephone, site_id, actif, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([
        $nom, 
        $prenom, 
        $email, 
        $password, 
        $role, 
        $telephone, 
        $site_id,
        $actif
    ]);
    
    if (!$result) {
        throw new Exception('Erreur lors de l\'insertion en base');
    }
    
    $userId = $pdo->lastInsertId();
    
    if (!$userId) {
        throw new Exception('Impossible de récupérer l\'ID du nouvel utilisateur');
    }
    
    // Récupérer l'utilisateur créé (sans le mot de passe)
    $getUserSql = "SELECT u.id, u.nom, u.prenom, u.email, u.role, u.telephone, u.actif, u.created_at,
                          s.nom_site, s.region
                   FROM utilisateurs u 
                   LEFT JOIN sites s ON u.site_id = s.id
                   WHERE u.id = ?";
    $getUserStmt = $pdo->prepare($getUserSql);
    $getUserStmt->execute([$userId]);
    $newUser = $getUserStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$newUser) {
        throw new Exception('Utilisateur créé mais impossible de le récupérer');
    }
    
    // Formater la réponse
    $newUser['actif'] = intval($newUser['actif']);
    $newUser['site_nom'] = $newUser['nom_site'];
    
    if ($newUser['nom_site'] && $newUser['region']) {
        $newUser['site_info'] = $newUser['nom_site'] . ' (' . $newUser['region'] . ')';
    } else {
        $newUser['site_info'] = null;
    }
    
    // Nettoyer les champs temporaires
    unset($newUser['nom_site'], $newUser['region']);
    
    echo json_encode([
        'success' => true,
        'message' => 'Utilisateur créé avec succès',
        'data' => $newUser,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (PDOException $e) {
    error_log("Erreur PDO: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur de base de données: ' . $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
} catch (Exception $e) {
    error_log("Erreur générale: " . $e->getMessage());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>