<?php
// api/users/list.php - Liste des utilisateurs avec gestion des permissions
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    // Connexion à la base de données
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Impossible de se connecter à la base de données');
    }
    
    // Récupérer l'utilisateur authentifié
    $headers = getallheaders();
    $authUser = null;
    
    if (isset($headers['Authorization'])) {
        $token = str_replace('Bearer ', '', $headers['Authorization']);
        try {
            $user_data = base64_decode($token);
            $authUser = json_decode($user_data, true);
        } catch (Exception $e) {
            // Token invalide, continuer sans restrictions pour l'instant
        }
    }
    
    // Construire la requête selon les permissions
    $sql = "SELECT 
                u.id, 
                u.nom, 
                u.prenom, 
                u.email, 
                u.telephone,
                u.role, 
                u.site_id,
                u.actif,
                u.created_at,
                s.nom_site,
                s.region
            FROM utilisateurs u 
            LEFT JOIN sites s ON u.site_id = s.id";
    
    $params = [];
    $whereConditions = [];
    
    // Appliquer les restrictions selon le rôle
    if ($authUser) {
        switch ($authUser['role']) {
            case 'controleur':
                // Le contrôleur ne voit que les utilisateurs de son site
                $whereConditions[] = "u.site_id = ?";
                $params[] = $authUser['site_id'];
                break;
                
            case 'superviseur':
                // Le superviseur voit tous les utilisateurs mais ne peut pas les modifier
                break;
                
            case 'admin':
                // L'admin voit tout
                break;
                
            case 'referent':
                // Le référent ne devrait pas accéder à cette API
                throw new Exception('Accès non autorisé pour ce rôle');
                
            default:
                throw new Exception('Rôle non reconnu');
        }
    }
    
    // Ajouter les conditions WHERE si nécessaire
    if (!empty($whereConditions)) {
        $sql .= " WHERE " . implode(" AND ", $whereConditions);
    }
    
    $sql .= " ORDER BY u.created_at DESC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Formater les données et ajouter les permissions
    foreach ($users as &$user) {
        // Convertir les valeurs numériques
        $user['id'] = (int)$user['id'];
        $user['site_id'] = $user['site_id'] ? (int)$user['site_id'] : null;
        $user['actif'] = (int)$user['actif'];
        
        // Formater les dates
        if ($user['created_at']) {
            $user['created_at'] = date('Y-m-d H:i:s', strtotime($user['created_at']));
        }
        
        // Ajouter les informations du site
        $user['site_info'] = null;
        if ($user['nom_site']) {
            $user['site_info'] = $user['nom_site'] . ' (' . $user['region'] . ')';
        }
        
        // Ajouter les permissions selon le rôle de l'utilisateur connecté
        $user['permissions'] = [
            'can_view' => true,
            'can_edit' => false,
            'can_delete' => false
        ];
        
        if ($authUser) {
            switch ($authUser['role']) {
                case 'admin':
                    $user['permissions']['can_edit'] = true;
                    $user['permissions']['can_delete'] = ($user['role'] !== 'admin'); // Ne peut pas supprimer un autre admin
                    break;
                    
                case 'superviseur':
                    // Le superviseur peut seulement voir
                    break;
                    
                case 'controleur':
                    // Le contrôleur peut modifier les référents de son site
                    if ($user['role'] === 'referent' && $user['site_id'] === $authUser['site_id']) {
                        $user['permissions']['can_edit'] = true;
                        $user['permissions']['can_delete'] = true;
                    }
                    break;
            }
        }
        
        // Gérer la photo (placeholder)
        $user['has_photo'] = false;
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Utilisateurs récupérés avec succès',
        'data' => $users,
        'count' => count($users),
        'user_role' => $authUser['role'] ?? 'unknown',
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (PDOException $e) {
    error_log("Erreur PDO users/list: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur de base de données: ' . $e->getMessage(),
        'data' => [],
        'timestamp' => date('Y-m-d H:i:s')
    ]);
} catch (Exception $e) {
    error_log("Erreur users/list: " . $e->getMessage());
    http_response_code(403);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'data' => [],
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>