<?php
// validation/validate.php - Validation des données par les contrôleurs
require_once '../config/database.php';
require_once '../config/auth_middleware.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(false, 'Méthode non autorisée', null, 405);
}

try {
    // Seuls admin et contrôleur peuvent valider
    $user = requireRole(['admin', 'controleur']);
    
    // Initialiser la connexion DB
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        jsonResponse(false, 'Erreur de connexion à la base de données', null, 500);
    }
    
    // Récupérer les données POST
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        jsonResponse(false, 'Données manquantes', null, 400);
    }
    
    $type = $input['type'] ?? ''; // 'profil' ou 'dechet'
    $id = intval($input['id'] ?? 0);
    $action = $input['action'] ?? ''; // 'validate' ou 'reject'
    $commentaire = $input['commentaire'] ?? '';
    
    if (!$type || !$id || !$action) {
        jsonResponse(false, 'Paramètres manquants (type, id, action)', null, 400);
    }
    
    if (!in_array($type, ['profil', 'dechet'])) {
        jsonResponse(false, 'Type invalide', null, 400);
    }
    
    if (!in_array($action, ['validate', 'reject'])) {
        jsonResponse(false, 'Action invalide', null, 400);
    }
    
    // Déterminer la table et vérifier les permissions
    $table = ($type === 'profil') ? 'profils_topographiques' : 'releves_dechets';
    
    // Récupérer l'élément à valider
    $stmt = $conn->prepare("SELECT * FROM $table WHERE id = ?");
    $stmt->execute([$id]);
    $item = $stmt->fetch();
    
    if (!$item) {
        jsonResponse(false, 'Élément non trouvé', null, 404);
    }
    
    // Vérifier les permissions
    if (!canValidateData($user, $item['site_id'])) {
        jsonResponse(false, 'Vous ne pouvez pas valider les données de ce site', null, 403);
    }
    
    // Vérifier que l'élément est en attente de validation
    if ($item['statut'] !== 'soumis') {
        jsonResponse(false, 'Seuls les éléments soumis peuvent être validés', null, 400);
    }
    
    // Déterminer le nouveau statut
    $nouveau_statut = ($action === 'validate') ? 'valide' : 'rejete';
    
    // Mettre à jour le statut
    $stmt = $conn->prepare("UPDATE $table SET statut = ? WHERE id = ?");
    $success = $stmt->execute([$nouveau_statut, $id]);
    
    if (!$success) {
        jsonResponse(false, 'Erreur lors de la mise à jour', null, 500);
    }
    
    // Enregistrer l'action dans un log (optionnel)
    try {
        $stmt = $conn->prepare("
            INSERT INTO validation_logs (type, item_id, validateur_id, action, commentaire, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$type, $id, $user['id'], $action, $commentaire]);
    } catch (Exception $e) {
        // Log silencieux si la table n'existe pas
        error_log("Log validation non enregistré: " . $e->getMessage());
    }
    
    $message = ($action === 'validate') ? 
        ucfirst($type) . ' validé avec succès' : 
        ucfirst($type) . ' rejeté avec succès';
    
    jsonResponse(true, $message, [
        'id' => $id,
        'type' => $type,
        'nouveau_statut' => $nouveau_statut,
        'validateur' => $user['nom'] . ' ' . $user['prenom']
    ]);
    
} catch (Exception $e) {
    error_log("Erreur validation/validate.php: " . $e->getMessage());
    jsonResponse(false, 'Erreur lors de la validation: ' . $e->getMessage(), null, 500);
}
?>